<?php
/**
 * CMS: LaiCMS (v1.0 Edition 2026)
 * File: blog/view.php
 * Description: Просмотр статьи с категориями, адаптивным шерингом и Glass-комментариями.
 */

require_once '../system/db.php';
require_once '../system/functions.php';

$slug = $_GET['slug'] ?? '';

// 1. Получаем статью, данные автора и КАТЕГОРИЮ
$stmt = $mysqli->prepare("
    SELECT p.*, u.username, u.avatar, c.name as cat_name, c.id as cat_id 
    FROM posts p 
    JOIN users u ON p.author_id = u.id 
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.slug = ? 
    LIMIT 1
");
$stmt->bind_param("s", $slug);
$stmt->execute();
$post = $stmt->get_result()->fetch_assoc();

if (!$post) {
    die("Статья не найдена.");
}

// 2. Логика удаления комментария
if (isset($_GET['delete_comment']) && isset($_SESSION['user_id'])) {
    $comment_id = (int)$_GET['delete_comment'];
    if (isAdmin() || $_SESSION['user_id'] == $post['author_id']) {
        $del = $mysqli->prepare("DELETE FROM comments WHERE id = ? AND post_id = ?");
        $del->bind_param("ii", $comment_id, $post['id']);
        if ($del->execute()) set_flash("Комментарий удален", "success");
    }
    header("Location: view.php?slug=" . $post['slug'] . "#comments");
    exit;
}

// 3. Обработка отправки комментария
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_comment']) && isset($_SESSION['user_id'])) {
    if (!isset($_POST['csrf_token']) || !check_csrf($_POST['csrf_token'])) die("Security error");
    $comment_text = trim($_POST['comment_text']);
    if (mb_strlen($comment_text) >= 2) {
        $ins = $mysqli->prepare("INSERT INTO comments (post_id, user_id, comment_text) VALUES (?, ?, ?)");
        $ins->bind_param("iis", $post['id'], $_SESSION['user_id'], $comment_text);
        $ins->execute();
        set_flash("Ваше мнение услышано!", "success");
        header("Location: view.php?slug=" . $post['slug'] . "#comments");
        exit;
    }
}

// 4. Получаем список комментариев
$comm_res = $mysqli->prepare("SELECT c.*, u.username, u.avatar FROM comments c JOIN users u ON c.user_id = u.id WHERE c.post_id = ? ORDER BY c.created_at DESC");
$comm_res->bind_param("i", $post['id']);
$comm_res->execute();
$comments = $comm_res->get_result();

$page_title = $post['title'];
include '../system/header.php';

$current_url = (isset($_SERVER['HTTPS']) ? "https" : "http") . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";
?>

<style>
    .article-container { max-width: 850px; margin: 0 auto; }
    .post-body { line-height: 1.8; font-size: 1.15rem; word-wrap: break-word; color: var(--pico-color); }
    .post-body img { max-width: 100%; height: auto; border-radius: 16px; margin: 2rem 0; box-shadow: 0 10px 30px rgba(0,0,0,0.1); }
    .post-body iframe { width: 100% !important; aspect-ratio: 16/9; border-radius: 16px; margin: 2rem 0; }
    
    .cat-badge { 
        display: inline-block; background: var(--pico-primary-background); 
        color: var(--pico-primary); padding: 4px 12px; border-radius: 50px; 
        font-size: 0.8rem; font-weight: bold; text-transform: uppercase; margin-bottom: 1rem;
    }

    /* ШЕРИНГ */
    .share-card { 
        margin: 4rem 0; padding: 2rem; border-radius: 20px;
        background: var(--pico-card-sectioning-background-color); 
        border: 1px dashed var(--pico-muted-border-color); text-align: center;
    }
    .share-buttons { display: flex; flex-wrap: wrap; gap: 12px; justify-content: center; margin-top: 1.5rem; }
    .share-btn { 
        padding: 10px 20px; border-radius: 50px; font-size: 0.9rem; font-weight: 600;
        color: white !important; text-decoration: none !important; transition: 0.3s;
        display: flex; align-items: center; gap: 8px; border: none;
    }
    .btn-vk { background: #0077FF; }
    .btn-tg { background: #229ED9; }
    .btn-copy { background: var(--pico-contrast); cursor: pointer; }
    .share-btn:hover { transform: scale(1.05); filter: brightness(1.1); }

    /* КОММЕНТАРИИ */
    .comment-item { 
        background: var(--pico-card-background-color); border-radius: 16px;
        padding: 1.5rem; margin-bottom: 1.5rem; border: 1px solid var(--pico-muted-border-color);
    }
    .comment-avatar { width: 45px; height: 45px; border-radius: 50%; object-fit: cover; }
    
    @media (max-width: 600px) {
        .post-body { font-size: 1.1rem; }
        .share-btn { width: 100%; justify-content: center; }
    }
</style>

<div class="article-container">
    <article>
        <header style="text-align: center; margin-bottom: 3rem;">
            <?php if($post['cat_name']): ?>
                <a href="index.php?cat=<?= $post['cat_id'] ?>" class="cat-badge" style="text-decoration: none;">
                    <?= _e($post['cat_name']) ?>
                </a>
            <?php endif; ?>
            
            <h1 style="font-size: 2.5rem; line-height: 1.2; margin-bottom: 1.5rem;"><?= _e($post['title']) ?></h1>
            
            <div style="display: flex; align-items: center; justify-content: center; gap: 15px;">
                <?php if($post['avatar']): ?>
                    <img src="<?= $post['avatar'] ?>" class="comment-avatar" style="border: 2px solid var(--pico-primary);">
                <?php endif; ?>
                <div style="text-align: left;">
                    <span style="display: block; font-weight: bold;"><?= _e($post['username']) ?></span>
                    <small class="secondary"><?= date('j F Y', strtotime($post['created_at'])) ?></small>
                </div>
            </div>
        </header>

        <div class="post-body">
            <?= $post['content'] ?>
        </div>

        <div class="share-card">
            <h5 style="margin:0;">Понравился материал?</h5>
            <p class="secondary" style="font-size: 0.9rem;">Поделитесь с друзьями или скопируйте ссылку</p>
            <div class="share-buttons">
                <a href="https://vk.com/share.php?url=<?= urlencode($current_url) ?>" target="_blank" class="share-btn btn-vk">
                    <i class="fa-brands fa-vk"></i> VK
                </a>
                <a href="https://t.me/share/url?url=<?= urlencode($current_url) ?>&text=<?= urlencode($post['title']) ?>" target="_blank" class="share-btn btn-tg">
                    <i class="fa-brands fa-telegram"></i> Telegram
                </a>
                <div onclick="copyLink(this)" class="share-btn btn-copy">
                    <i class="fa-solid fa-link"></i> <span>Ссылка</span>
                </div>
            </div>
        </div>

        <footer style="background:transparent; border-top: 1px solid var(--pico-muted-border-color); padding-top: 2rem;">
            <div class="grid">
                <a href="index.php" class="secondary outline" role="button">← Назад в ленту</a>
                <?php if(isset($_SESSION['user_id']) && ($_SESSION['user_id'] == $post['author_id'] || isAdmin())): ?>
                    <a href="/blog/edit_post.php?id=<?= $post['id'] ?>" class="contrast">Редактировать статью</a>
                <?php endif; ?>
            </div>
        </footer>
    </article>

    <section id="comments" style="margin-top: 4rem;">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
            <h3 style="margin:0;"><i class="fa-regular fa-comments"></i> Комментарии <small style="opacity:0.5;"><?= $comments->num_rows ?></small></h3>
        </div>

        <?php if(isset($_SESSION['user_id'])): ?>
            <div class="comment-item" style="border-style: dashed; border-color: var(--pico-primary);">
                <form method="POST" style="margin: 0;">
                    <input type="hidden" name="csrf_token" value="<?= _e($_SESSION['csrf_token']) ?>">
                    <textarea name="comment_text" placeholder="Напишите, что вы думаете..." style="margin-bottom: 1rem; min-height: 100px;" required></textarea>
                    <button type="submit" name="add_comment" class="primary" style="width: auto; padding: 10px 30px;">Отправить</button>
                </form>
            </div>
        <?php else: ?>
            <div style="text-align: center; padding: 2rem; border: 1px solid var(--pico-muted-border-color); border-radius: 16px; margin-bottom: 2rem;">
                <p class="secondary">Хотите принять участие в обсуждении?</p>
                <a href="/users/login.php" role="button" class="outline">Войти в аккаунт</a>
            </div>
        <?php endif; ?>

        <div class="comment-list">
            <?php while($c = $comments->fetch_assoc()): ?>
                <div class="comment-item">
                    <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 1rem;">
                        <div style="display: flex; align-items: center; gap: 12px;">
                            <?php if($c['avatar']): ?>
                                <img src="<?= $c['avatar'] ?>" class="comment-avatar">
                            <?php else: ?>
                                <div class="comment-avatar" style="background: var(--pico-primary); color: white; display: flex; align-items: center; justify-content: center; font-weight: bold; font-size: 0.9rem;">
                                    <?= mb_substr($c['username'], 0, 1) ?>
                                </div>
                            <?php endif; ?>
                            <div>
                                <strong style="display: block; line-height: 1;"><?= _e($c['username']) ?></strong>
                                <small class="secondary" style="font-size: 0.7rem;"><?= date('d.m.Y H:i', strtotime($c['created_at'])) ?></small>
                            </div>
                        </div>
                        <?php if(isset($_SESSION['user_id']) && (isAdmin() || $_SESSION['user_id'] == $post['author_id'])): ?>
                            <a href="?slug=<?= $slug ?>&delete_comment=<?= $c['id'] ?>" class="secondary" onclick="return confirm('Удалить этот комментарий?')" title="Удалить">
                                <i class="fa-solid fa-xmark"></i>
                            </a>
                        <?php endif; ?>
                    </div>
                    <div style="font-size: 1rem; line-height: 1.6; color: var(--pico-color);">
                        <?= nl2br(_e($c['comment_text'])) ?>
                    </div>
                </div>
            <?php endwhile; ?>
            
            <?php if($comments->num_rows === 0): ?>
                <div style="text-align: center; padding: 3rem; opacity: 0.5;">
                    <i class="fa-solid fa-feather-pointed fa-2x" style="margin-bottom: 1rem; display: block;"></i>
                    <p>Пока никто не оставил отзыв. Будьте первым!</p>
                </div>
            <?php endif; ?>
        </div>
    </section>
</div>

<script>
function copyLink(el) {
    const textSpan = el.querySelector('span');
    const originalText = textSpan.innerText;
    
    navigator.clipboard.writeText('<?= $current_url ?>').then(() => {
        el.style.background = '#2ecc71';
        textSpan.innerText = 'Скопировано!';
        setTimeout(() => {
            el.style.background = '';
            textSpan.innerText = originalText;
        }, 2000);
    }).catch(err => {
        alert('Ошибка при копировании');
    });
}
</script>

<?php include '../system/footer.php'; ?>

