<?php
/**
 * CMS: LaiCMS (v1.0 Edition 2026)
 * File: blog/my_posts.php
 * Description: Кабинет автора с расширенной аналитикой и управлением категориями.
 */

require_once '../system/db.php';
require_once '../system/functions.php';

// 1. ПРОВЕРКА АВТОРИЗАЦИИ
if (!isset($_SESSION['user_id'])) {
    header("Location: /users/login.php");
    exit;
}

$user_id = (int)$_SESSION['user_id'];

// 2. БЕЗОПАСНОЕ УДАЛЕНИЕ
if (isset($_GET['delete']) && isset($_GET['token'])) {
    if ($_GET['token'] === $_SESSION['csrf_token']) {
        $post_id = (int)$_GET['delete'];
        
        $del_stmt = $mysqli->prepare("DELETE FROM posts WHERE id = ? AND author_id = ?");
        $del_stmt->bind_param("ii", $post_id, $user_id);
        
        if ($del_stmt->execute() && $mysqli->affected_rows > 0) {
            if (file_exists('../seo_gen.php')) {
                ob_start(); include '../seo_gen.php'; ob_end_clean();
            }
            set_flash("Статья безвозвратно удалена", "success");
        } else {
            set_flash("Ошибка доступа: это не ваш контент", "error");
        }
    } else {
        set_flash("Ошибка безопасности (Invalid CSRF)", "error");
    }
    header("Location: my_posts.php");
    exit;
}

// 3. ПОЛУЧЕНИЕ ДАННЫХ С КАТЕГОРИЯМИ
$stmt = $mysqli->prepare("
    SELECT p.id, p.title, p.slug, p.created_at, c.name as cat_name,
    (SELECT COUNT(*) FROM comments WHERE post_id = p.id) as comm_count
    FROM posts p 
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.author_id = ? 
    ORDER BY p.id DESC
");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$my_posts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$posts_count = count($my_posts);

// Считаем общее число комментариев для автора
$total_comments = array_sum(array_column($my_posts, 'comm_count'));

$page_title = "Мои статьи — LaiCMS";
include '../system/header.php';
?>

<style>
    .author-dashboard { margin-top: 1rem; }
    .stat-box { 
        padding: 1.5rem; border-radius: 20px; 
        background: var(--pico-card-background-color); 
        border: 1px solid var(--pico-muted-border-color);
        transition: transform 0.2s ease;
    }
    .stat-box:hover { transform: translateY(-3px); border-color: var(--pico-primary); }
    .post-table-card { border-radius: 20px; overflow: hidden; margin-top: 2rem; border: 1px solid var(--pico-muted-border-color); }
    
    .cat-mini { 
        font-size: 0.65rem; background: var(--pico-primary-background); 
        color: var(--pico-primary); padding: 2px 8px; border-radius: 4px; 
        font-weight: bold; text-transform: uppercase;
    }
    
    .action-group { display: flex; gap: 8px; justify-content: flex-end; }
    .btn-circle { width: 38px; height: 38px; display: flex; align-items: center; justify-content: center; border-radius: 50%; transition: 0.2s; }
    .btn-circle:hover { filter: brightness(1.1); transform: scale(1.1); }

    @media (max-width: 600px) { .hide-mobile { display: none; } }
</style>

<div class="author-dashboard">
    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem; flex-wrap: wrap; gap: 1rem;">
        <hgroup style="margin: 0;">
            <h1 style="margin:0;"><i class="fa-solid fa-id-badge" style="color: var(--pico-primary);"></i> Студия автора</h1>
            <p class="secondary">Добро пожаловать, <strong><?= _e($_SESSION['username']) ?></strong></p>
        </hgroup>
        <a href="add_post.php" role="button" class="primary" style="border-radius: 50px;">
            <i class="fa-solid fa-plus-circle"></i> Новая запись
        </a>
    </div>

    <?php display_flash(); ?>

    <div class="grid">
        <div class="stat-box">
            <small class="secondary"><i class="fa-solid fa-file-lines"></i> ПУБЛИКАЦИИ</small>
            <h2 style="margin:0;"><?= $posts_count ?></h2>
            <progress value="<?= $posts_count ?>" max="100" style="margin-top: 10px;"></progress>
        </div>
        <div class="stat-box">
            <small class="secondary"><i class="fa-solid fa-comments"></i> ОБСУЖДЕНИЯ</small>
            <h2 style="margin:0;"><?= $total_comments ?></h2>
            <p style="font-size: 0.7rem; margin:0;" class="secondary">Всего отзывов на ваши посты</p>
        </div>
        <div class="stat-box">
            <small class="secondary"><i class="fa-solid fa-shield-halved"></i> СТАТУС</small>
            <h2 style="margin:0; font-size: 1.2rem;"><?= isAdmin() ? 'ADMIN' : 'AUTHOR' ?></h2>
            <p style="font-size: 0.7rem; margin:0; color: #2ecc71;">Аккаунт подтвержден</p>
        </div>
    </div>

    <article class="post-table-card">
        <header style="background: var(--pico-card-sectioning-background-color); border-bottom: 1px solid var(--pico-muted-border-color);">
            <h5 style="margin:0;">Управление контентом</h5>
        </header>
        <div style="overflow-x: auto;">
            <table role="grid" style="margin-bottom: 0;">
                <thead>
                    <tr>
                        <th>Статья</th>
                        <th class="hide-mobile">Категория</th>
                        <th style="text-align: center;"><i class="fa-regular fa-comments"></i></th>
                        <th style="text-align: right;">Действия</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($posts_count > 0): ?>
                        <?php foreach ($my_posts as $post): ?>
                        <tr>
                            <td>
                                <div style="font-weight: 600; line-height: 1.2;">
                                    <a href="view.php?slug=<?= $post['slug'] ?>" class="contrast"><?= _e($post['title']) ?></a>
                                </div>
                                <small class="secondary" style="font-size: 0.7rem;"><?= date('d M Y', strtotime($post['created_at'])) ?></small>
                            </td>
                            <td class="hide-mobile">
                                <span class="cat-mini"><?= $post['cat_name'] ?? 'Без категории' ?></span>
                            </td>
                            <td style="text-align: center;">
                                <kbd style="font-size: 0.7rem;"><?= $post['comm_count'] ?></kbd>
                            </td>
                            <td>
                                <div class="action-group">
                                    <a href="edit_post.php?id=<?= $post['id'] ?>" class="btn-circle" style="background: var(--pico-primary-background); color: var(--pico-primary);" title="Редактировать">
                                        <i class="fa-solid fa-pen"></i>
                                    </a>
                                    <a href="?delete=<?= $post['id'] ?>&token=<?= $_SESSION['csrf_token'] ?>" 
                                       class="btn-circle" style="background: #fff5f5; color: #e74c3c; border: 1px solid #ffe3e3;"
                                       onclick="return confirm('Вы уверены? Статью нельзя будет восстановить.')">
                                        <i class="fa-solid fa-trash"></i>
                                    </a>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="4" style="text-align: center; padding: 4rem;">
                                <div style="opacity: 0.3; margin-bottom: 1rem;"><i class="fa-solid fa-layer-group fa-3x"></i></div>
                                <p class="secondary">У вас пока нет публикаций. Время заявить о себе!</p>
                                <a href="add_post.php" class="outline primary" role="button">Написать первый пост</a>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </article>
</div>

<footer style="margin-top: 3rem; text-align: center;">
    <a href="index.php" class="secondary outline" style="border-radius: 50px;">
        <i class="fa-solid fa-arrow-left"></i> Назад в общий блог
    </a>
</footer>

<?php include '../system/footer.php'; ?>

