// =========================
// app.js – основной кастомный JS для сайта
// =========================

document.addEventListener('DOMContentLoaded', () => {

    // =========================
    // Универсальный AJAX POST с CSRF
    // =========================
    window.ajaxPost = async (url, data) => {
        try {
            const response = await fetch(url, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data)
            });
            return await response.json();
        } catch (err) {
            console.error('AJAX error:', err);
            return { success: false, error: 'Ошибка соединения с сервером' };
        }
    };

    // =========================
    // Уведомления (alert)
    // =========================
    const showAlert = (message, type = 'success', timeout = 5000) => {
        const alertContainer = document.querySelector('#alert-container');
        if (!alertContainer) return;
        const alert = document.createElement('div');
        alert.className = `alert alert-${type} fade-in`;
        alert.textContent = message;
        alertContainer.appendChild(alert);
        setTimeout(() => alert.remove(), timeout);
    };

    // =========================
    // Загрузка фото через AJAX
    // =========================
    const photoForm = document.querySelector('#photo-upload-form');
    if (photoForm) {
        photoForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = new FormData(photoForm);
            formData.append('csrf', window.CSRF_TOKEN);

            const response = await fetch('/ajax/upload_photo.php', {
                method: 'POST',
                body: formData
            });
            const res = await response.json();
            if (res.success) {
                showAlert('Фото успешно загружено!');
                // Добавляем превью
                const previewContainer = document.querySelector('#photo-preview');
                if (previewContainer) {
                    const img = document.createElement('img');
                    img.src = res.url;
                    img.className = 'img-fluid rounded slide-up';
                    previewContainer.appendChild(img);
                }
            } else {
                showAlert(res.error || 'Ошибка загрузки фото', 'danger');
            }
        });
    }

    // =========================
    // Корзина: добавление / удаление / обновление
    // =========================
    document.querySelectorAll('.btn-add-to-cart').forEach(btn => {
        btn.addEventListener('click', async () => {
            const itemKey = btn.dataset.key;
            const res = await ajaxPost('/ajax/cart_add.php', { item_key: itemKey, csrf: window.CSRF_TOKEN });
            showAlert(res.success ? 'Товар добавлен в корзину' : res.error, res.success ? 'success' : 'danger');
        });
    });

    document.querySelectorAll('.btn-remove-from-cart').forEach(btn => {
        btn.addEventListener('click', async () => {
            const itemKey = btn.dataset.key;
            const res = await ajaxPost('/ajax/cart_remove.php', { item_key: itemKey, csrf: window.CSRF_TOKEN });
            showAlert(res.success ? 'Товар удалён из корзины' : res.error, res.success ? 'success' : 'danger');
            if (res.success) btn.closest('tr').remove();
        });
    });

    document.querySelectorAll('.cart-quantity').forEach(input => {
        input.addEventListener('change', async () => {
            const itemKey = input.dataset.key;
            const quantity = parseInt(input.value);
            if (quantity <= 0) return;
            const res = await ajaxPost('/ajax/cart_update.php', { item_key: itemKey, quantity, csrf: window.CSRF_TOKEN });
            showAlert(res.success ? 'Корзина обновлена' : res.error, res.success ? 'success' : 'danger');
        });
    });

    // =========================
    // Применение скидки
    // =========================
    const discountForm = document.querySelector('#discount-form');
    if (discountForm) {
        discountForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const discount = discountForm.querySelector('input[name="discount"]').value;
            const res = await ajaxPost('/ajax/apply_discount.php', { discount, csrf: window.CSRF_TOKEN });
            if (res.success) {
                showAlert(`Скидка ${res.discount_percent}% применена!`);
                const totalEl = document.querySelector('#cart-total');
                if (totalEl) totalEl.textContent = res.discounted_total.toFixed(2);
            } else {
                showAlert(res.error || 'Неверная скидка', 'danger');
            }
        });
    }

    // =========================
    // Checkout AJAX
    // =========================
    const checkoutForm = document.querySelector('#checkout-form');
    if (checkoutForm) {
        checkoutForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const formData = {};
            checkoutForm.querySelectorAll('input, select').forEach(el => formData[el.name] = el.value);
            formData.csrf = window.CSRF_TOKEN;

            const res = await ajaxPost('/ajax/checkout.php', formData);
            if (res.success) {
                showAlert('Заказ успешно оформлен!');
                document.querySelector('#cart-items').innerHTML = '';
                const totalEl = document.querySelector('#cart-total');
                if (totalEl) totalEl.textContent = '0.00';
            } else {
                showAlert(res.error || 'Ошибка оформления заказа', 'danger');
            }
        });
    }

});
