# Найкращі практики

Рекомендації для розробки на FLOWAXY-CMS.

## Загальні принципи

### 1. Дотримуйтесь стандартів коду

- Використовуйте `declare(strict_types=1);`
- Дотримуйтесь PSR-12
- Використовуйте типізацію
- Документуйте код

### 2. Безпека

- Завжди використовуйте CSRF захист
- Санитуйте всі вхідні дані
- Валідуйте дані перед збереженням
- Хешуйте паролі

### 3. Продуктивність

- Використовуйте кешування
- Оптимізуйте запити до БД
- Мінімізуйте кількість HTTP запитів
- Використовуйте версіонування assets

## Розробка плагінів

### 1. Структура

```
plugin-slug/
├── Plugin.php
├── admin/
│   └── pages/
│       └── plugin-page.php
├── config/
│   └── plugin.ini
└── README.md
```

### 2. Реєстрація хуків

```php
// Добре: Через методи плагіна
public function registerHooks(): void
{
    $this->registerAction('init', [$this, 'onInit']);
}

// Погано: Глобальні функції
add_action('init', function() { });
```

### 3. Налаштування

```php
// Добре: Через методи плагіна
$settings = $this->getSettings();
$this->setSetting('key', 'value');

// Погано: Прямий доступ до файлів
$settings = parse_ini_file('config.ini');
```

## Розробка тем

### 1. Структура

```
theme-slug/
├── Theme.php
├── components/
├── widgets/
├── layouts/
├── templates/
└── resources/
```

### 2. Компоненти

```php
// Добре: Використання компонентів
<?php theme_component('slider'); ?>

// Погано: Дублювання коду
<div class="slider">...</div>
```

### 3. Assets

```php
// Добре: Версіонування
<link rel="stylesheet" href="<?php echo theme_asset('css/style.css'); ?>">

// Погано: Без версіонування
<link rel="stylesheet" href="/css/style.css">
```

## Робота з БД

### 1. Використання QueryBuilder

```php
// Добре: QueryBuilder
$users = Database::table('users')
    ->where('status', 'active')
    ->get();

// Погано: Прямі SQL запити
$users = Database::select("SELECT * FROM users WHERE status = 'active'");
```

### 2. Підготовлені запити

```php
// Добре: Підготовлені запити
$users = Database::select('SELECT * FROM users WHERE status = ?', ['active']);

// Погано: Конкатенація рядків
$users = Database::select("SELECT * FROM users WHERE status = '{$status}'");
```

## Робота з хуками

### 1. Пріоритети

```php
// Добре: Правильні пріоритети
Action::add('init', $callback1, 5);  // Високий
Action::add('init', $callback2, 10); // Стандартний
Action::add('init', $callback3, 20); // Низький

// Погано: Всі з однаковим пріоритетом
Action::add('init', $callback1, 10);
Action::add('init', $callback2, 10);
```

### 2. Назви хуків

```php
// Добре: Унікальні назви
Action::add('my_plugin_init', $callback);

// Погано: Загальні назви
Action::add('init', $callback);
```

## Безпека

### 1. CSRF захист

```php
// Добре: CSRF захист
<form method="POST">
    <?php echo csrf_token_field(); ?>
    <!-- Поля -->
</form>

// Погано: Без CSRF захисту
<form method="POST">
    <!-- Поля -->
</form>
```

### 2. Санитизація

```php
// Добре: Санитизація
$clean = sanitize_input($_POST['input'] ?? '');

// Погано: Без санитизації
$input = $_POST['input'];
```

### 3. Екранування виводу

```php
// Добре: htmlspecialchars
echo htmlspecialchars($value, ENT_QUOTES, 'UTF-8');

// Погано: Без екранування
echo $value;
```

## Продуктивність

### 1. Кешування

```php
// Добре: Кешування
$value = Cache::remember('key', function() {
    return expensive_operation();
}, 3600);

// Погано: Без кешування
$value = expensive_operation();
```

### 2. Оптимізація запитів

```php
// Добре: Один запит
$users = Database::table('users')->get();

// Погано: Багато запитів
foreach ($ids as $id) {
    $user = Database::table('users')->where('id', $id)->first();
}
```

## Наступні кроки

- [Швидкий старт](Quick-Start) — швидке керівництво
- [API Reference](../API-Reference/Facades) — повний API довідник

---

**Дотримання найкращих практик забезпечує якісний код!** ✨
