# Віджети тем

Детальний опис створення та використання віджетів у темах.

## Огляд

Віджети — це блоки контенту з адаптивними стилями, які можна використовувати для відображення динамічного контенту.

## Структура віджета

```
widgets/[widget-name]/
├── widget.php        # HTML та PHP логіка
├── widget.css        # Стилі віджета
└── responsive.css    # Адаптивні стилі
```

## Створення віджета

### Крок 1: Створення директорії

Створіть директорію для віджета:

```
content/themes/my-theme/widgets/marketplace/
```

### Крок 2: Створення widget.php

```php
<?php
/**
 * Marketplace Widget
 * Віджет магазину
 */

// Дані віджета
$items = $items ?? [
    ['name' => 'Item 1', 'icon' => 'fas fa-box'],
    ['name' => 'Item 2', 'icon' => 'fas fa-cube'],
];

$title = $title ?? 'Магазин';
$description = $description ?? '';
?>
<div class="marketplace-widget">
    <div class="marketplace-header">
        <h3 class="marketplace-title"><?php echo htmlspecialchars($title, ENT_QUOTES, 'UTF-8'); ?></h3>
        <?php if (!empty($description)): ?>
            <p class="marketplace-description"><?php echo htmlspecialchars($description, ENT_QUOTES, 'UTF-8'); ?></p>
        <?php endif; ?>
    </div>
    <div class="marketplace-grid">
        <?php foreach ($items as $item): ?>
            <div class="marketplace-item">
                <?php if (!empty($item['icon'])): ?>
                    <div class="marketplace-icon">
                        <i class="<?php echo htmlspecialchars($item['icon'], ENT_QUOTES, 'UTF-8'); ?>"></i>
                    </div>
                <?php endif; ?>
                <div class="marketplace-name"><?php echo htmlspecialchars($item['name'] ?? '', ENT_QUOTES, 'UTF-8'); ?></div>
            </div>
        <?php endforeach; ?>
    </div>
</div>
```

### Крок 3: Створення widget.css

```css
.marketplace-widget {
    padding: 28px;
    margin-bottom: 24px;
    background: #fff;
    border: 1px solid #d0d0d0;
}

.marketplace-widget .marketplace-header {
    margin-bottom: 24px;
    border-bottom: 1px solid #e8e8e8;
    padding-bottom: 16px;
}

.marketplace-widget .marketplace-title {
    font-size: 20px;
    font-weight: 700;
    color: #1a1a1a;
    text-transform: uppercase;
    letter-spacing: 1px;
    margin: 0 0 8px 0;
}

.marketplace-widget .marketplace-description {
    font-size: 14px;
    color: #4a4a4a;
    margin: 0;
}

.marketplace-widget .marketplace-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
    gap: 12px;
}

.marketplace-widget .marketplace-item {
    background: #f8f8f8;
    border: 1px solid #d0d0d0;
    padding: 24px 16px;
    text-align: center;
    transition: background-color 0.2s, border-color 0.2s;
}

.marketplace-widget .marketplace-item:hover {
    background: #f0f0f0;
    border-color: #1a1a1a;
}

.marketplace-widget .marketplace-icon {
    font-size: 40px;
    color: #4a4a4a;
    margin-bottom: 12px;
}

.marketplace-widget .marketplace-name {
    font-size: 12px;
    color: #1a1a1a;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}
```

### Крок 4: Створення responsive.css

```css
/* Планшети */
@media (max-width: 1024px) {
    .marketplace-widget {
        padding: 20px;
    }

    .marketplace-widget .marketplace-grid {
        grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
        gap: 10px;
    }
}

/* Мобільні */
@media (max-width: 768px) {
    .marketplace-widget {
        padding: 16px;
        margin-bottom: 16px;
    }

    .marketplace-widget .marketplace-title {
        font-size: 18px;
    }

    .marketplace-widget .marketplace-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 8px;
    }

    .marketplace-widget .marketplace-item {
        padding: 16px 12px;
    }

    .marketplace-widget .marketplace-icon {
        font-size: 32px;
    }
}
```

## Використання віджета

### Базове використання

```php
<?php theme_widget('marketplace'); ?>
```

### З параметрами

```php
<?php
theme_widget('marketplace', [
    'title' => 'Магазин плагінів',
    'description' => 'Розширення та теми для Flowaxy CMS',
    'items' => [
        ['name' => 'Плагіни', 'icon' => 'fas fa-puzzle-piece'],
        ['name' => 'Темі', 'icon' => 'fas fa-palette'],
        ['name' => 'Віджети', 'icon' => 'fas fa-cube'],
    ],
]);
?>
```

## Автоматичне підключення стилів

Система автоматично підключає `widget.css` та `responsive.css` при виклику віджета:

```php
<?php theme_widget('marketplace'); ?>
```

Автоматично підключаються:
- `widgets/marketplace/widget.css`
- `widgets/marketplace/responsive.css`

## Версіонування ресурсів

Система автоматично додає версію до CSS файлів:

```html
<link rel="stylesheet" href="/content/themes/my-theme/widgets/marketplace/widget.css?v=1234567890">
<link rel="stylesheet" href="/content/themes/my-theme/widgets/marketplace/responsive.css?v=1234567890">
```

## Ізоляція даних

Віджети автоматично ізолюють дані через `extract()`:

```php
// У widget.php
extract($widgetData, EXTR_OVERWRITE);

// Тепер $title, $items доступні безпосередньо
echo $title;
foreach ($items as $item) { }
```

## Найкращі практики

### 1. Адаптивний дизайн

```css
/* Завжди створюйте responsive.css */
@media (max-width: 768px) {
    .marketplace-widget .marketplace-grid {
        grid-template-columns: repeat(2, 1fr);
    }
}
```

### 2. Ізоляція стилів

```css
/* Добре: Префікс для ізоляції */
.marketplace-widget { }
.marketplace-widget .item { }

/* Погано: Глобальні стилі */
.item { }
```

### 3. Валідація даних

```php
// Добре: Перевірка та значення за замовчуванням
$items = $items ?? [];
$title = $title ?? 'Віджет';

// Погано: Без перевірки
foreach ($items as $item) { }
```

### 4. Екранування виводу

```php
// Добре: htmlspecialchars
echo htmlspecialchars($title, ENT_QUOTES, 'UTF-8');

// Погано: Без екранування
echo $title;
```

## Приклади віджетів

### Marketplace Widget

Віджет магазину плагінів та тем.

**Структура:**
```
widgets/marketplace/
├── widget.php
├── widget.css
└── responsive.css
```

### Portfolio Widget

Віджет портфоліо проектів.

**Структура:**
```
widgets/portfolio/
├── widget.php
├── widget.css
└── responsive.css
```

### Blogs Widget

Віджет блогів.

**Структура:**
```
widgets/blogs/
├── widget.php
├── widget.css
└── responsive.css
```

## Наступні кроки

- [Створення теми](Creating-Theme) — покрокове створення
- [Компоненти](Components) — створення компонентів
- [Макети та шаблони](Layouts-Templates) — робота з макетами

---

**Віджети забезпечують гнучкість та адаптивність!** 📦
