# Структура теми

Детальний опис структури теми та призначення кожного елемента.

## Загальна структура

```
content/themes/[theme-name]/
├── Theme.php            # Клас теми (обов'язково)
├── components/          # Компоненти теми
├── widgets/             # Віджети теми
├── layouts/             # Макети теми
├── templates/           # Шаблони теми
├── resources/           # Ресурси теми
│   ├── styles/          # CSS файли
│   ├── scripts/         # JavaScript файли
│   ├── fonts/           # Шрифти
│   └── library/         # Бібліотеки
└── customization.php    # Налаштування кастомізації
```

## Theme.php

**Призначення:** Основний клас теми, який успадковує `BaseTheme`.

**Обов'язкові методи:**
- `renderTheme()` — рендеринг теми
- `registerHooks()` — реєстрація хуків (опціонально)

**Приклад:**
```php
final class Theme extends BaseTheme
{
    public function registerHooks(): void
    {
        add_theme_support('widgets');
        add_theme_support('menus', ['header', 'footer']);
    }

    public function renderTheme(): void
    {
        $this->render('default');
    }
}
```

## components/

**Призначення:** Переіспользувані блоки з власними стилями та скриптами.

**Структура компонента:**
```
components/[component-name]/
├── component.php    # HTML та PHP логіка
├── component.css    # Стилі компонента
└── component.js     # JavaScript компонента
```

**Використання:**
```php
<?php theme_component('slider'); ?>
```

**Детальніше:** [Компоненти](Components)

## widgets/

**Призначення:** Блоки контенту з адаптивними стилями.

**Структура віджета:**
```
widgets/[widget-name]/
├── widget.php        # HTML та PHP логіка
├── widget.css        # Стилі віджета
└── responsive.css    # Адаптивні стилі
```

**Використання:**
```php
<?php theme_widget('marketplace', $data); ?>
```

**Детальніше:** [Віджети](Widgets)

## layouts/

**Призначення:** Загальна структура сторінок (header, footer, sidebars).

**Структура макету:**
```
layouts/
└── main.php    # Основний макет
```

**Приклад:**
```php
<!DOCTYPE html>
<html>
<head>
    <title><?php echo $pageTitle; ?></title>
</head>
<body>
    <header>...</header>
    <main><?php echo Theme::yield('content'); ?></main>
    <footer>...</footer>
</body>
</html>
```

**Детальніше:** [Макети та шаблони](Layouts-Templates)

## templates/

**Призначення:** Конкретні сторінки, які використовують макети.

**Структура шаблону:**
```
templates/
├── default.php    # Головна сторінка
├── single.php     # Одна стаття
└── archive.php    # Архів
```

**Приклад:**
```php
<?php Theme::extend('main', ['content' => function() { ?>
    <h1>Головна сторінка</h1>
<?php }]); ?>
```

**Детальніше:** [Макети та шаблони](Layouts-Templates)

## resources/

**Призначення:** Статичні ресурси теми.

### resources/styles/

CSS файли теми.

**Приклад:**
```
resources/styles/
├── style.css         # Основний стиль
└── responsive.css    # Адаптивні стилі
```

**Підключення:**
```php
<link rel="stylesheet" href="<?php echo theme_asset('styles/style.css'); ?>">
```

### resources/scripts/

JavaScript файли теми.

**Приклад:**
```
resources/scripts/
└── main.js    # Основний скрипт
```

**Підключення:**
```php
<script src="<?php echo theme_asset('scripts/main.js'); ?>"></script>
```

### resources/fonts/

Шрифти теми.

**Приклад:**
```
resources/fonts/
├── raleway-regular.woff2
└── kanit-bold.woff2
```

**Використання:**
```css
@font-face {
    font-family: 'Raleway';
    src: url('../fonts/raleway-regular.woff2') format('woff2');
}
```

### resources/library/

Бібліотеки (jQuery, Font Awesome, тощо).

**Приклад:**
```
resources/library/
├── jquery-3.7.1.min.js
└── font-awesome/
    └── css/
        └── all.min.css
```

## customization.php

**Призначення:** Налаштування кастомізації теми для адмін-панелі.

**Приклад:**
```php
<?php
return [
    'colors' => [
        'primary' => [
            'label' => 'Основний колір',
            'type' => 'color',
            'default' => '#1a1a1a',
        ],
        'secondary' => [
            'label' => 'Додатковий колір',
            'type' => 'color',
            'default' => '#4a4a4a',
        ],
    ],
    'typography' => [
        'font_family' => [
            'label' => 'Шрифт',
            'type' => 'select',
            'options' => [
                'Arial' => 'Arial',
                'Helvetica' => 'Helvetica',
            ],
            'default' => 'Arial',
        ],
    ],
];
?>
```

## Автоматичне завантаження

### Компоненти

Компоненти автоматично завантажуються при виклику:

```php
<?php theme_component('slider'); ?>
```

Система шукає файл `components/slider/component.php`.

### Віджети

Віджети автоматично завантажуються при виклику:

```php
<?php theme_widget('marketplace', $data); ?>
```

Система шукає файл `widgets/marketplace/widget.php`.

### Макети та шаблони

Макети та шаблони завантажуються через `Theme::extend()` та `Theme::render()`:

```php
Theme::extend('main', ['content' => function() { /* ... */ }]);
Theme::render('default');
```

## Версіонування ресурсів

Система автоматично додає версію до ресурсів для cache busting:

```php
theme_asset('styles/style.css');
// Результат: /content/themes/my-theme/resources/styles/style.css?v=1234567890
```

Версія генерується на основі `filemtime()` файлу.

## Найкращі практики

### 1. Організація файлів

```
✅ Добре: Логічна структура
components/
├── slider/
├── auth/
└── social-networks/

❌ Погано: Без структури
components/
├── slider.php
├── auth.php
└── social.php
```

### 2. Ізоляція стилів

```css
/* Добре: Префікс для ізоляції */
.slider-component { }
.slider-component .slide { }

/* Погано: Глобальні стилі */
.slide { }
```

### 3. Використання змінних

```php
// Добре: Змінні для конфігурації
$slides = $slides ?? [];
$autoplay = $autoplay ?? true;

// Погано: Хардкод
$slides = [['title' => 'Slide 1']];
```

## Наступні кроки

- [Створення теми](Creating-Theme) — покрокове створення
- [Компоненти](Components) — створення компонентів
- [Віджети](Widgets) — створення віджетів
- [Макети та шаблони](Layouts-Templates) — робота з макетами

---

**Структура теми забезпечує організацію та підтримуваність!** 📁
