# Огляд системи тем

Загальний огляд системи тем FLOWAXY-CMS та їх архітектури.

## Що таке тема?

Тема — це інтерфейс користувача, який визначає зовнішній вигляд та функціональність сайту. Тема може містити компоненти, віджети, макети та шаблони.

## Переваги системи тем

- **Гнучкість** — повний контроль над дизайном
- **Модульність** — компоненти та віджети для переіспользування
- **Розширюваність** — інтеграція з плагінами через хуки
- **Адаптивність** — вбудована підтримка responsive дизайну
- **Кастомізація** — налаштування через адмін-панель

## Архітектура теми

### Базовий клас

Всі темі успадковують `BaseTheme`:

```php
<?php
namespace Flowaxy\Themes\MyTheme;

use Flowaxy\Support\Base\BaseTheme;

final class Theme extends BaseTheme
{
    public function registerHooks(): void
    {
        // Реєстрація хуків
    }

    public function renderTheme(): void
    {
        $this->render('default');
    }
}
```

### Структура теми

```
content/themes/[theme-name]/
├── Theme.php            # Клас теми (обов'язково)
├── components/          # Компоненти теми
│   └── [component]/
│       ├── component.php
│       ├── component.css
│       └── component.js
├── widgets/             # Віджети теми
│   └── [widget]/
│       ├── widget.php
│       ├── widget.css
│       └── responsive.css
├── layouts/             # Макети теми
│   └── main.php
├── templates/           # Шаблони теми
│   └── default.php
├── resources/           # Ресурси теми
│   ├── styles/          # CSS файли
│   ├── scripts/         # JS файли
│   ├── fonts/           # Шрифти
│   └── library/         # Бібліотеки
└── customization.php    # Налаштування кастомізації
```

## Компоненти теми

Компоненти — це переіспользувані блоки з власними стилями та скриптами.

**Структура:**
```
components/[component-name]/
├── component.php    # HTML та PHP логіка
├── component.css    # Стилі компонента
└── component.js     # JavaScript компонента
```

**Використання:**
```php
<?php theme_component('slider'); ?>
```

**Детальніше:** [Компоненти](Components)

## Віджети теми

Віджети — це блоки контенту з адаптивними стилями.

**Структура:**
```
widgets/[widget-name]/
├── widget.php        # HTML та PHP логіка
├── widget.css        # Стилі віджета
└── responsive.css    # Адаптивні стилі
```

**Використання:**
```php
<?php theme_widget('marketplace', $data); ?>
```

**Детальніше:** [Віджети](Widgets)

## Макети та шаблони

### Макети (Layouts)

Макети визначають загальну структуру сторінок (header, footer, sidebars).

**Приклад:**
```php
// layouts/main.php
<!DOCTYPE html>
<html>
<head>
    <title><?php echo $pageTitle; ?></title>
</head>
<body>
    <?php echo Theme::yield('content'); ?>
</body>
</html>
```

### Шаблони (Templates)

Шаблони визначають конкретні сторінки, використовуючи макети.

**Приклад:**
```php
// templates/default.php
<?php Theme::extend('main', ['content' => function() { ?>
    <h1>Головна сторінка</h1>
<?php }]); ?>
```

**Детальніше:** [Макети та шаблони](Layouts-Templates)

## Реєстрація хуків

Темі можуть реєструвати хуки для інтеграції з системою:

```php
public function registerHooks(): void
{
    // Підтримка можливостей теми
    add_theme_support('widgets');
    add_theme_support('menus', ['header', 'footer']);

    // Реєстрація областей меню
    register_nav_menu('header', 'Header Menu');
    register_nav_menu('footer', 'Footer Menu');
}
```

## Налаштування теми

Темі можуть зберігати налаштування:

```php
// Отримання налаштувань
$settings = $this->getSettings();

// Збереження налаштування
$this->setSetting('key', 'value');
```

Налаштування зберігаються в `storage/config/themes/{theme-slug}.ini`.

## Адмін-панель для теми

Темі можуть створювати сторінки в адмін-панелі:

```php
private array $adminThemePages = [
    'theme-navigation' => [
        'menu' => ['text' => 'Навігація', 'icon' => 'fas fa-bars'],
        'callback' => 'admin_theme_navigation_page',
        'file' => 'theme-navigation.php',
    ],
];
```

## Приклади тем

### Default Theme

Повнофункціональна тема за замовчуванням з компонентами, віджетами та адаптивним дизайном.

**Детальніше:** [Приклади тем](Examples)

## Наступні кроки

- [Створення теми](Creating-Theme) — покрокове створення
- [Структура теми](Theme-Structure) — детальний опис структури
- [Компоненти](Components) — створення компонентів
- [Віджети](Widgets) — створення віджетів
- [Макети та шаблони](Layouts-Templates) — робота з макетами
- [Приклади](Examples) — готові приклади тем

---

**Система тем забезпечує повний контроль над дизайном!** 🎨
