# Макети та шаблони

Детальний опис роботи з макетами та шаблонами в темах.

## Огляд

Макети визначають загальну структуру сторінок (header, footer, sidebars), а шаблони визначають конкретний контент сторінок.

## Макети (Layouts)

### Призначення

Макети містять загальну структуру HTML, яка повторюється на всіх сторінках.

### Створення макету

Створіть файл `layouts/main.php`:

```php
<!DOCTYPE html>
<html lang="uk">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle ?? 'Сайт'; ?></title>
    <link rel="stylesheet" href="<?php echo theme_asset('styles/style.css'); ?>">
</head>
<body>
    <header>
        <!-- Header content -->
    </header>

    <main>
        <?php echo Theme::yield('content'); ?>
    </main>

    <footer>
        <!-- Footer content -->
    </footer>

    <script src="<?php echo theme_asset('scripts/main.js'); ?>"></script>
</body>
</html>
```

### Використання yield()

`Theme::yield()` використовується для вставки контенту з шаблонів:

```php
<?php echo Theme::yield('content'); ?>
```

Можна використовувати кілька секцій:

```php
<?php echo Theme::yield('header'); ?>
<?php echo Theme::yield('content'); ?>
<?php echo Theme::yield('sidebar'); ?>
<?php echo Theme::yield('footer'); ?>
```

## Шаблони (Templates)

### Призначення

Шаблони визначають конкретний контент сторінок, використовуючи макети.

### Створення шаблону

Створіть файл `templates/default.php`:

```php
<?php
use Flowaxy\Support\Theme\Theme;

Theme::extend('main', [
    'content' => function() {
        ?>
        <h1>Головна сторінка</h1>
        <p>Вітаємо на сайті!</p>
        <?php
    }
]);
?>
```

### Використання extend()

`Theme::extend()` використовується для розширення макету:

```php
Theme::extend('main', [
    'content' => function() {
        // Контент сторінки
    }
]);
```

### Кілька секцій

```php
Theme::extend('main', [
    'header' => function() {
        ?>
        <div class="page-header">
            <h1>Заголовок сторінки</h1>
        </div>
        <?php
    },
    'content' => function() {
        ?>
        <div class="page-content">
            <p>Контент сторінки</p>
        </div>
        <?php
    },
    'sidebar' => function() {
        ?>
        <div class="page-sidebar">
            <h3>Бічна панель</h3>
        </div>
        <?php
    },
]);
```

## Повний приклад

### Макет (layouts/main.php)

```php
<!DOCTYPE html>
<html lang="uk">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle ?? 'Сайт'; ?></title>
    <link rel="stylesheet" href="<?php echo theme_asset('styles/style.css'); ?>">
</head>
<body>
    <header class="site-header">
        <div class="header-container">
            <div class="logo">
                <a href="/"><?php echo htmlspecialchars($siteName ?? 'Flowaxy', ENT_QUOTES, 'UTF-8'); ?></a>
            </div>
            <nav class="header-nav">
                <?php
                $headerMenu = $headerMenu ?? [];
                foreach ($headerMenu as $item):
                ?>
                    <a href="<?php echo htmlspecialchars($item['url'] ?? '#', ENT_QUOTES, 'UTF-8'); ?>">
                        <?php echo htmlspecialchars($item['name'] ?? '', ENT_QUOTES, 'UTF-8'); ?>
                    </a>
                <?php endforeach; ?>
            </nav>
        </div>
    </header>

    <div class="main-layout">
        <aside class="sidebar-left">
            <?php echo Theme::yield('sidebar-left'); ?>
        </aside>

        <main class="main-content">
            <?php echo Theme::yield('content'); ?>
        </main>

        <aside class="sidebar-right">
            <?php echo Theme::yield('sidebar-right'); ?>
        </aside>
    </div>

    <footer class="site-footer">
        <div class="footer-container">
            <?php echo Theme::yield('footer'); ?>
        </div>
    </footer>

    <script src="<?php echo theme_asset('scripts/main.js'); ?>"></script>
</body>
</html>
```

### Шаблон (templates/default.php)

```php
<?php
use Flowaxy\Support\Theme\Theme;

Theme::extend('main', [
    'sidebar-left' => function() {
        ?>
        <div class="sidebar-section">
            <h3>Навігація</h3>
            <ul>
                <li><a href="/">Головна</a></li>
                <li><a href="/about">Про нас</a></li>
                <li><a href="/contact">Контакти</a></li>
            </ul>
        </div>
        <?php
    },
    'content' => function() {
        ?>
        <div class="page-content">
            <?php theme_component('slider'); ?>

            <section class="widgets-section">
                <?php theme_widget('marketplace', ['items' => $marketplaceItems ?? []]); ?>
                <?php theme_widget('portfolio', ['items' => $portfolioItems ?? []]); ?>
            </section>
        </div>
        <?php
    },
    'sidebar-right' => function() {
        ?>
        <div class="sidebar-section">
            <?php theme_component('auth'); ?>
        </div>
        <?php
    },
    'footer' => function() {
        ?>
        <div class="footer-content">
            <p>&copy; <?php echo date('Y'); ?> <?php echo htmlspecialchars($_SERVER['HTTP_HOST'] ?? 'flowaxy.com', ENT_QUOTES, 'UTF-8'); ?></p>
            <?php theme_component('social-networks'); ?>
        </div>
        <?php
    },
]);
?>
```

## Змінні в макетах

Макети мають доступ до змінних, переданих з шаблонів:

```php
// У шаблоні
Theme::extend('main', [
    'content' => function() use ($pageData) {
        echo $pageData['title'];
    }
]);

// У макеті
<?php echo $pageTitle ?? 'Сайт'; ?>
```

## Найкращі практики

### 1. Використовуйте значення за замовчуванням

```php
// Добре: Значення за замовчуванням
$pageTitle = $pageTitle ?? 'Сайт';
$siteName = $siteName ?? 'Flowaxy';

// Погано: Без перевірки
echo $pageTitle; // Може бути undefined
```

### 2. Екранування виводу

```php
// Добре: htmlspecialchars
echo htmlspecialchars($title, ENT_QUOTES, 'UTF-8');

// Погано: Без екранування
echo $title;
```

### 3. Організація секцій

```php
// Добре: Логічні секції
Theme::extend('main', [
    'header' => function() { },
    'content' => function() { },
    'footer' => function() { },
]);

// Погано: Все в одній секції
Theme::extend('main', [
    'content' => function() {
        // Header, content, footer все разом
    },
]);
```

### 4. Використання компонентів та віджетів

```php
// Добре: Використання компонентів
<?php theme_component('slider'); ?>
<?php theme_widget('marketplace', $data); ?>

// Погано: Дублювання коду
<div class="slider">...</div>
```

## Наступні кроки

- [Створення теми](Creating-Theme) — покрокове створення
- [Компоненти](Components) — створення компонентів
- [Віджети](Widgets) — створення віджетів

---

**Макети та шаблони забезпечують структуру та організацію!** 📄
