# Створення теми

Покрокове керівництво з створення теми для FLOWAXY-CMS.

## Крок 1: Створення структури

### Створення директорії

Створіть директорію для теми в `content/themes/`:

```
content/themes/my-theme/
```

### Базова структура

```
content/themes/my-theme/
├── Theme.php           # Клас теми (обов'язково)
├── components/         # Компоненти (опціонально)
├── widgets/            # Віджети (опціонально)
├── layouts/            # Макети (опціонально)
├── templates/          # Шаблони (опціонально)
├── resources/          # Ресурси (опціонально)
│   ├── styles/
│   ├── scripts/
│   ├── fonts/
│   └── library/
└── customization.php   # Налаштування (опціонально)
```

## Крок 2: Створення класу теми

### Базовий клас

Створіть файл `Theme.php`:

```php
<?php

/**
 * My Theme
 *
 * @name: My Theme
 * @slug: my-theme
 * @version: 1.0.0
 * @description: Опис теми
 * @author: Ваше ім'я
 * @author_url: https://example.com
 * @requires: 1.0.0
 * @tested: 1.0.0
 * @package: MyTheme
 *
 * FLOWAXY CMS 1.0.0 Stable
 */

declare(strict_types=1);

use Flowaxy\Support\Base\BaseTheme;

final class Theme extends BaseTheme
{
    public function registerHooks(): void
    {
        // Реєстрація хуків
    }

    public function renderTheme(): void
    {
        $this->render('default');
    }
}
```

### Метадані теми

Метадані вказуються в PHPDoc блоці на початку файлу:

- `@name` — назва теми
- `@slug` — унікальний ідентифікатор
- `@version` — версія теми
- `@description` — опис функціональності
- `@author` — автор теми
- `@author_url` — URL автора
- `@requires` — мінімальна версія FLOWAXY-CMS
- `@tested` — версія, на якій протестовано
- `@package` — namespace пакету

## Крок 3: Реєстрація хуків

### Підтримка можливостей

```php
public function registerHooks(): void
{
    // Підтримка віджетів
    add_theme_support('widgets');

    // Підтримка меню
    add_theme_support('menus', ['header', 'footer']);
}
```

### Реєстрація областей меню

```php
public function registerHooks(): void
{
    // Реєстрація областей меню
    register_nav_menu('header', 'Header Menu');
    register_nav_menu('footer', 'Footer Menu');
}
```

## Крок 4: Створення макету

### Базовий макет

Створіть файл `layouts/main.php`:

```php
<!DOCTYPE html>
<html lang="uk">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle ?? 'Сайт'; ?></title>
    <link rel="stylesheet" href="<?php echo theme_asset('styles/style.css'); ?>">
</head>
<body>
    <header>
        <!-- Header content -->
    </header>

    <main>
        <?php echo Theme::yield('content'); ?>
    </main>

    <footer>
        <!-- Footer content -->
    </footer>

    <script src="<?php echo theme_asset('scripts/main.js'); ?>"></script>
</body>
</html>
```

## Крок 5: Створення шаблону

### Базовий шаблон

Створіть файл `templates/default.php`:

```php
<?php
use Flowaxy\Support\Theme\Theme;

Theme::extend('main', [
    'content' => function() {
        ?>
        <h1>Головна сторінка</h1>
        <p>Вітаємо на сайті!</p>
        <?php
    }
]);
?>
```

## Крок 6: Створення компонента

### Структура компонента

Створіть директорію `components/slider/`:

```
components/slider/
├── component.php
├── component.css
└── component.js
```

### component.php

```php
<?php
$slides = $slides ?? [
    ['title' => 'Slide 1', 'description' => 'Description 1'],
    ['title' => 'Slide 2', 'description' => 'Description 2'],
];
?>
<div class="slider-component">
    <?php foreach ($slides as $slide): ?>
        <div class="slide">
            <h2><?php echo htmlspecialchars($slide['title']); ?></h2>
            <p><?php echo htmlspecialchars($slide['description']); ?></p>
        </div>
    <?php endforeach; ?>
</div>
```

### Використання компонента

```php
<?php theme_component('slider'); ?>
```

## Крок 7: Створення віджета

### Структура віджета

Створіть директорію `widgets/marketplace/`:

```
widgets/marketplace/
├── widget.php
├── widget.css
└── responsive.css
```

### widget.php

```php
<?php
$items = $items ?? [];
?>
<div class="marketplace-widget">
    <h3>Магазин</h3>
    <div class="marketplace-grid">
        <?php foreach ($items as $item): ?>
            <div class="marketplace-item">
                <h4><?php echo htmlspecialchars($item['name']); ?></h4>
            </div>
        <?php endforeach; ?>
    </div>
</div>
```

### Використання віджета

```php
<?php theme_widget('marketplace', ['items' => $data]); ?>
```

## Крок 8: Додавання ресурсів

### CSS файли

Створіть файл `resources/styles/style.css`:

```css
body {
    font-family: Arial, sans-serif;
    margin: 0;
    padding: 0;
}

header {
    background: #333;
    color: #fff;
    padding: 20px;
}
```

### JavaScript файли

Створіть файл `resources/scripts/main.js`:

```javascript
document.addEventListener('DOMContentLoaded', function() {
    console.log('Theme loaded');
});
```

### Підключення ресурсів

```php
// У макеті
<link rel="stylesheet" href="<?php echo theme_asset('styles/style.css'); ?>">
<script src="<?php echo theme_asset('scripts/main.js'); ?>"></script>
```

## Крок 9: Тестування

### Перевірка встановлення

1. Перейдіть в адмін-панель → **Темі**
2. Знайдіть вашу тему в списку
3. Натисніть **Встановити**

### Перевірка активації

1. Після встановлення натисніть **Активувати**
2. Перевірте відображення теми на сайті
3. Перевірте роботу компонентів та віджетів

## Повний приклад

```php
<?php

/**
 * Example Theme
 *
 * @name: Example Theme
 * @slug: example-theme
 * @version: 1.0.0
 * @description: Приклад теми
 * @author: Flowaxy Team
 * @author_url: https://flowaxy.com
 * @requires: 1.0.0
 * @tested: 1.0.0
 * @package: ExampleTheme
 */

declare(strict_types=1);

use Flowaxy\Support\Base\BaseTheme;

final class Theme extends BaseTheme
{
    public function registerHooks(): void
    {
        // Підтримка можливостей
        add_theme_support('widgets');
        add_theme_support('menus', ['header', 'footer']);

        // Реєстрація меню
        register_nav_menu('header', 'Header Menu');
        register_nav_menu('footer', 'Footer Menu');
    }

    public function renderTheme(): void
    {
        $this->render('default');
    }
}
```

## Наступні кроки

- [Структура теми](Theme-Structure) — детальний опис структури
- [Компоненти](Components) — створення компонентів
- [Віджети](Widgets) — створення віджетів
- [Макети та шаблони](Layouts-Templates) — робота з макетами
- [Приклади](Examples) — готові приклади тем

---

**Тема створено! Тепер можна налаштовувати дизайн!** 🎨
