# Компоненти тем

Детальний опис створення та використання компонентів у темах.

## Огляд

Компоненти — це переіспользувані блоки з власними стилями та скриптами, які можна використовувати в будь-якому місці теми.

## Структура компонента

```
components/[component-name]/
├── component.php    # HTML та PHP логіка
├── component.css    # Стилі компонента
└── component.js     # JavaScript компонента
```

## Створення компонента

### Крок 1: Створення директорії

Створіть директорію для компонента:

```
content/themes/my-theme/components/slider/
```

### Крок 2: Створення component.php

```php
<?php
/**
 * Slider Component
 * Компонент слайдера
 */

// Параметри за замовчуванням
$slides = $slides ?? [
    [
        'title' => 'Slide 1',
        'description' => 'Description 1',
        'button' => 'Learn More',
    ],
    [
        'title' => 'Slide 2',
        'description' => 'Description 2',
        'button' => 'Get Started',
    ],
];

$autoplay = $autoplay ?? true;
$interval = $interval ?? 5000;
?>
<div class="slider-component" data-autoplay="<?php echo $autoplay ? 'true' : 'false'; ?>" data-interval="<?php echo $interval; ?>">
    <div class="slider-wrapper">
        <?php foreach ($slides as $index => $slide): ?>
            <div class="slider-slide <?php echo $index === 0 ? 'active' : ''; ?>" data-slide="<?php echo $index; ?>">
                <div class="slide-content">
                    <h2 class="slide-title"><?php echo htmlspecialchars($slide['title'] ?? '', ENT_QUOTES, 'UTF-8'); ?></h2>
                    <p class="slide-description"><?php echo htmlspecialchars($slide['description'] ?? '', ENT_QUOTES, 'UTF-8'); ?></p>
                    <?php if (!empty($slide['button'])): ?>
                        <button class="slide-btn"><?php echo htmlspecialchars($slide['button'], ENT_QUOTES, 'UTF-8'); ?></button>
                    <?php endif; ?>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
    <div class="slider-controls">
        <button class="slider-btn slider-btn-prev">‹</button>
        <button class="slider-btn slider-btn-next">›</button>
    </div>
    <div class="slider-dots">
        <?php foreach ($slides as $index => $slide): ?>
            <button class="slider-dot <?php echo $index === 0 ? 'active' : ''; ?>" data-slide="<?php echo $index; ?>"></button>
        <?php endforeach; ?>
    </div>
</div>
```

### Крок 3: Створення component.css

```css
.slider-component {
    position: relative;
    width: 100%;
    height: 400px;
    overflow: hidden;
}

.slider-component .slider-wrapper {
    display: flex;
    transition: transform 0.5s ease;
}

.slider-component .slider-slide {
    min-width: 100%;
    height: 400px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: #f0f0f0;
}

.slider-component .slider-slide.active {
    display: flex;
}

.slider-component .slide-content {
    text-align: center;
    padding: 20px;
}

.slider-component .slide-title {
    font-size: 32px;
    margin-bottom: 16px;
}

.slider-component .slide-description {
    font-size: 16px;
    margin-bottom: 24px;
}

.slider-component .slide-btn {
    padding: 12px 32px;
    background: #1a1a1a;
    color: #fff;
    border: none;
    cursor: pointer;
}

.slider-component .slider-controls {
    position: absolute;
    top: 50%;
    transform: translateY(-50%);
    width: 100%;
    display: flex;
    justify-content: space-between;
    padding: 0 20px;
}

.slider-component .slider-btn {
    background: rgba(255, 255, 255, 0.5);
    border: none;
    padding: 10px 15px;
    cursor: pointer;
}

.slider-component .slider-dots {
    position: absolute;
    bottom: 20px;
    left: 50%;
    transform: translateX(-50%);
    display: flex;
    gap: 8px;
}

.slider-component .slider-dot {
    width: 12px;
    height: 12px;
    border-radius: 50%;
    border: none;
    background: rgba(255, 255, 255, 0.5);
    cursor: pointer;
}

.slider-component .slider-dot.active {
    background: #fff;
}
```

### Крок 4: Створення component.js

```javascript
(function() {
    const sliders = document.querySelectorAll('.slider-component');

    sliders.forEach(slider => {
        const wrapper = slider.querySelector('.slider-wrapper');
        const slides = slider.querySelectorAll('.slider-slide');
        const prevBtn = slider.querySelector('.slider-btn-prev');
        const nextBtn = slider.querySelector('.slider-btn-next');
        const dots = slider.querySelectorAll('.slider-dot');
        const autoplay = slider.dataset.autoplay === 'true';
        const interval = parseInt(slider.dataset.interval) || 5000;

        let currentSlide = 0;
        let autoplayTimer = null;

        function showSlide(index) {
            slides.forEach((slide, i) => {
                slide.classList.toggle('active', i === index);
            });
            dots.forEach((dot, i) => {
                dot.classList.toggle('active', i === index);
            });
            wrapper.style.transform = `translateX(-${index * 100}%)`;
            currentSlide = index;
        }

        function nextSlide() {
            const next = (currentSlide + 1) % slides.length;
            showSlide(next);
        }

        function prevSlide() {
            const prev = (currentSlide - 1 + slides.length) % slides.length;
            showSlide(prev);
        }

        if (prevBtn) prevBtn.addEventListener('click', prevSlide);
        if (nextBtn) nextBtn.addEventListener('click', nextSlide);

        dots.forEach((dot, index) => {
            dot.addEventListener('click', () => showSlide(index));
        });

        if (autoplay) {
            function startAutoplay() {
                autoplayTimer = setInterval(nextSlide, interval);
            }

            function stopAutoplay() {
                if (autoplayTimer) {
                    clearInterval(autoplayTimer);
                    autoplayTimer = null;
                }
            }

            slider.addEventListener('mouseenter', stopAutoplay);
            slider.addEventListener('mouseleave', startAutoplay);

            startAutoplay();
        }
    });
})();
```

## Використання компонента

### Базове використання

```php
<?php theme_component('slider'); ?>
```

### З параметрами

```php
<?php
theme_component('slider', [
    'slides' => [
        ['title' => 'Slide 1', 'description' => 'Desc 1'],
        ['title' => 'Slide 2', 'description' => 'Desc 2'],
    ],
    'autoplay' => true,
    'interval' => 3000,
]);
?>
```

## Автоматичне підключення стилів та скриптів

Система автоматично підключає `component.css` та `component.js` при виклику компонента:

```php
<?php theme_component('slider'); ?>
```

Автоматично підключаються:
- `components/slider/component.css`
- `components/slider/component.js`

## Версіонування ресурсів

Система автоматично додає версію до CSS та JS файлів:

```html
<link rel="stylesheet" href="/content/themes/my-theme/components/slider/component.css?v=1234567890">
<script src="/content/themes/my-theme/components/slider/component.js?v=1234567890"></script>
```

## Найкращі практики

### 1. Ізоляція стилів

```css
/* Добре: Префікс для ізоляції */
.slider-component { }
.slider-component .slide { }

/* Погано: Глобальні стилі */
.slide { }
```

### 2. Валідація даних

```php
// Добре: Перевірка та значення за замовчуванням
$slides = $slides ?? [];
$autoplay = $autoplay ?? true;

// Погано: Без перевірки
foreach ($slides as $slide) { }
```

### 3. Екранування виводу

```php
// Добре: htmlspecialchars
echo htmlspecialchars($title, ENT_QUOTES, 'UTF-8');

// Погано: Без екранування
echo $title;
```

### 4. Документація

```php
<?php
/**
 * Slider Component
 *
 * @param array $slides Масив слайдів
 * @param bool $autoplay Автопрогравання
 * @param int $interval Інтервал автопрогравання (мс)
 */
?>
```

## Приклади компонентів

### Auth Component

Компонент авторизації та реєстрації.

**Структура:**
```
components/auth/
├── component.php
├── component.css
└── component.js
```

### Social Networks Component

Компонент соціальних мереж (інтеграція з плагіном).

**Структура:**
```
components/social-networks/
├── component.php
└── component.css
```

## Наступні кроки

- [Створення теми](Creating-Theme) — покрокове створення
- [Віджети](Widgets) — створення віджетів
- [Макети та шаблони](Layouts-Templates) — робота з макетами

---

**Компоненти забезпечують переіспользуваність та модульність!** 🧩
