# Життєвий цикл плагіна

Детальний опис життєвого циклу плагіна та методів управління.

## Огляд життєвого циклу

Життєвий цикл плагіна складається з чотирьох основних стадій:

1. **Install** — встановлення
2. **Activate** — активація
3. **Deactivate** — деактивація
4. **Uninstall** — видалення

## Стадії життєвого циклу

### 1. Install (Встановлення)

**Коли викликається:** При першому встановленні плагіна через адмін-панель.

**Що робити:**
- Створення таблиць БД (якщо потрібно)
- Створення конфігураційних файлів
- Створення директорій
- Ініціалізація базових налаштувань

**Метод:**
```php
public function install(): void
{
    parent::install();

    // Створення таблиць БД
    $this->createDatabaseTables();

    // Створення конфігурацій
    $this->initializeConfig();
}
```

**Примітка:** Міграції БД виконуються при активації, а не при встановленні.

### 2. Activate (Активація)

**Коли викликається:** При активації плагіна через адмін-панель.

**Що робити:**
- Виконання міграцій БД
- Реєстрація хуків
- Ініціалізація даних
- Створення тимчасових файлів

**Метод:**
```php
public function activate(): void
{
    parent::activate();

    // Виконання міграцій
    $this->runMigrations();

    // Ініціалізація даних
    $this->initializeData();
}
```

**Примітка:** Хуки автоматично реєструються через `registerHooks()` при завантаженні плагіна.

### 3. Deactivate (Деактивація)

**Коли викликається:** При деактивації плагіна через адмін-панель.

**Що робити:**
- Очищення тимчасових даних
- Зупинка фонових процесів
- Збереження стану (опціонально)

**Метод:**
```php
public function deactivate(): void
{
    parent::deactivate();

    // Очищення тимчасових даних
    $this->clearTempData();

    // Зупинка процесів
    $this->stopBackgroundProcesses();
}
```

**Примітка:** Дані та налаштування залишаються, тільки функціональність вимикається.

### 4. Uninstall (Видалення)

**Коли викликається:** При видаленні плагіна через адмін-панель.

**Що робити:**
- Видалення таблиць БД
- Видалення конфігурацій
- Видалення файлів (опціонально)
- Очищення кешу

**Метод:**
```php
public function uninstall(): void
{
    parent::uninstall();

    // Видалення таблиць БД
    $this->dropDatabaseTables();

    // Видалення конфігурацій
    $this->removeConfig();
}
```

**Примітка:** Після видалення всі дані плагіна видаляються безповоротно.

## Статуси плагіна

Плагін може мати наступні статуси:

- **non-install** — не встановлений
- **installed** — встановлений, але не активований
- **activated** — встановлений та активований
- **deactivated** — встановлений, але деактивований

## Статуси зберігаються в `storage/config/system/plugins.ini`:

```ini
[plugins]
my-plugin = "activated"
another-plugin = "installed"
```

## Методи BasePlugin

### registerHooks()

Викликається автоматично при завантаженні плагіна:

```php
public function registerHooks(): void
{
    // Реєстрація хуків
    $this->registerAction('init', [$this, 'onInit']);
    $this->registerFilter('the_title', [$this, 'formatTitle']);
}
```

### registerRoutes()

Викликається автоматично при завантаженні плагіна:

```php
public function registerRoutes(): void
{
    $this->registerHook('admin_routes', function ($router) {
        $router->add(['GET', 'POST'], 'my-plugin', 'admin_my_plugin_page');
    });
}
```

### init()

Викликається автоматично при завантаженні плагіна:

```php
public function init(): void
{
    // Додаткова ініціалізація
    // Автоматично викликає registerHooks() та registerRoutes()
}
```

## Приклади використання

### Створення таблиць БД при встановленні

```php
public function install(): void
{
    parent::install();

    $db = DatabaseHelper::getConnection();
    $db->exec("
        CREATE TABLE IF NOT EXISTS my_plugin_data (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(255) NOT NULL,
            value TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
}
```

### Виконання міграцій при активації

```php
public function activate(): void
{
    parent::activate();

    // Виконання міграцій з папки migrations/
    $migrationsDir = $this->pluginDir . 'migrations';
    if (is_dir($migrationsDir)) {
        $this->runMigrations($migrationsDir);
    }
}
```

### Ініціалізація конфігурації при активації

```php
public function activate(): void
{
    parent::activate();

    // Створення конфігурації за замовчуванням
    $configFile = $this->pluginDir . 'config' . DS . 'settings.ini';
    if (!file_exists($configFile)) {
        $defaultConfig = [
            'setting1' => 'value1',
            'setting2' => 'value2',
        ];
        IniHelper::writeFile($configFile, $defaultConfig);
    }
}
```

### Очищення кешу при деактивації

```php
public function deactivate(): void
{
    parent::deactivate();

    // Очищення кешу плагіна
    CacheHelper::forget('my_plugin_cache');
}
```

### Видалення таблиць при видаленні

```php
public function uninstall(): void
{
    parent::uninstall();

    $db = DatabaseHelper::getConnection();
    $db->exec("DROP TABLE IF EXISTS my_plugin_data");
}
```

## Хуки життєвого циклу

Система надає хуки для відстеження життєвого циклу:

- `plugin_installed` — після встановлення
- `plugin_activated` — після активації
- `plugin_deactivated` — після деактивації
- `plugin_uninstalled` — після видалення

**Приклад використання:**

```php
// В іншому плагіні або темі
Action::add('plugin_activated', function($pluginSlug) {
    if ($pluginSlug === 'my-plugin') {
        // Логіка при активації my-plugin
    }
});
```

## Найкращі практики

### 1. Не видаляйте дані при деактивації

```php
// Добре: зберігаємо дані
public function deactivate(): void
{
    parent::deactivate();
    // Тільки очищення тимчасових даних
}

// Погано: видаляємо дані
public function deactivate(): void
{
    parent::deactivate();
    $this->dropDatabaseTables(); // Не робіть так!
}
```

### 2. Використовуйте транзакції для БД

```php
public function install(): void
{
    parent::install();

    $db = DatabaseHelper::getConnection();
    $db->beginTransaction();

    try {
        $db->exec("CREATE TABLE ...");
        $db->exec("INSERT INTO ...");
        $db->commit();
    } catch (Exception $e) {
        $db->rollBack();
        throw $e;
    }
}
```

### 3. Перевіряйте наявність перед створенням

```php
public function install(): void
{
    parent::install();

    $configFile = $this->pluginDir . 'config' . DS . 'settings.ini';
    if (!file_exists($configFile)) {
        // Створюємо тільки якщо не існує
        IniHelper::writeFile($configFile, $defaultConfig);
    }
}
```

## Наступні кроки

- [Створення плагіна](Creating-Plugin) — покрокове створення
- [Хуки для плагінів](Plugin-Hooks) — використання хуків
- [Адмін-панель](Plugin-Admin) — створення адмін-інтерфейсу

---

**Життєвий цикл плагіна забезпечує контроль над встановленням та видаленням!** 🔄
