# Хуки для плагінів

Детальний опис використання хуків (Actions та Filters) у плагінах.

## Огляд

Хуки дозволяють плагінам інтегруватися з системою та іншими плагінами без зміни ядра.

## Типи хуків

### Actions (Події)

Actions використовуються для виконання дій у певні моменти.

**Характеристики:**
- Не повертають значення
- Виконуються послідовно
- Можуть мати пріоритет

### Filters (Фільтри)

Filters використовуються для зміни даних перед їх використанням.

**Характеристики:**
- Повертають значення
- Можуть бути ланцюжковими
- Можуть мати пріоритет

## Реєстрація хуків

### Через методи BasePlugin

```php
public function registerHooks(): void
{
    // Реєстрація Action
    $this->registerAction('init', [$this, 'onInit']);

    // Реєстрація Filter
    $this->registerFilter('the_title', [$this, 'formatTitle']);
}
```

### Через замикання

```php
public function registerHooks(): void
{
    // Action через замикання
    $this->registerAction('admin_init', function() {
        // Логіка ініціалізації
    });

    // Filter через замикання
    $this->registerFilter('admin_menu', function(array $menu): array {
        $menu[] = ['text' => 'Мій плагін', 'href' => '/admin/my-plugin'];
        return $menu;
    });
}
```

### З пріоритетом

```php
public function registerHooks(): void
{
    // Високий пріоритет (виконується першим)
    $this->registerAction('init', [$this, 'onInit'], 5);

    // Низький пріоритет (виконується останнім)
    $this->registerFilter('the_content', [$this, 'processContent'], 20);
}
```

## Стандартні хуки системи

### Actions

#### init
Виконується при ініціалізації системи.

```php
$this->registerAction('init', function() {
    // Ініціалізація плагіна
});
```

#### plugins_loaded
Виконується після завантаження всіх плагінів.

```php
$this->registerAction('plugins_loaded', function() {
    // Логіка після завантаження плагінів
});
```

#### admin_init
Виконується при ініціалізації адмінки.

```php
$this->registerAction('admin_init', function() {
    // Ініціалізація адмінки
});
```

#### before_render
Виконується перед рендерингом сторінки.

```php
$this->registerAction('before_render', function() {
    // Логіка перед рендерингом
});
```

#### after_render
Виконується після рендерингу сторінки.

```php
$this->registerAction('after_render', function() {
    // Логіка після рендерингу
});
```

### Filters

#### admin_menu
Фільтрація пунктів меню адмінки.

```php
$this->registerFilter('admin_menu', function(array $menu): array {
    $menu[] = [
        'text' => 'Мій плагін',
        'href' => admin_url('my-plugin'),
        'page' => 'my-plugin',
        'icon' => 'fas fa-cog',
        'order' => 50,
    ];
    return $menu;
});
```

#### the_title
Фільтрація заголовка.

```php
$this->registerFilter('the_title', function(string $title): string {
    return strtoupper($title);
});
```

#### the_content
Фільтрація контенту.

```php
$this->registerFilter('the_content', function(string $content): string {
    return nl2br($content);
});
```

## Створення власних хуків

### Створення Action

```php
// Реєстрація обробника
$this->registerAction('my_custom_action', [$this, 'handleAction']);

// Виконання action (в іншому місці)
Action::do('my_custom_action', ['param1', 'param2']);
```

### Створення Filter

```php
// Реєстрація обробника
$this->registerFilter('my_custom_filter', function($value) {
    return $value . ' modified';
});

// Застосування filter
$result = Filter::apply('my_custom_filter', 'original');
// Результат: 'original modified'
```

## Приклади використання

### Додавання пункту в меню адмінки

```php
public function registerHooks(): void
{
    $this->registerFilter('admin_menu', function(array $menu): array {
        $menu[] = [
            'text' => 'Соціальні мережі',
            'title' => 'Управління соціальними мережами',
            'icon' => 'fas fa-share-alt',
            'href' => admin_url('social-networks'),
            'page' => 'social-networks',
            'permission' => 'admin.access',
            'order' => 46,
        ];
        return $menu;
    });
}
```

### Фільтрація даних для теми

```php
public function registerHooks(): void
{
    $this->registerFilter('theme_social_networks', function(array $networks): array {
        $settings = $this->getSettings();
        if (($settings['status'] ?? 'activated') === 'deactivated') {
            return $networks;
        }
        return array_merge($networks, $this->getSocialNetworks());
    });
}
```

### Обробка подій системи

```php
public function registerHooks(): void
{
    // При ініціалізації
    $this->registerAction('init', function() {
        // Завантаження конфігурації
        $this->loadConfig();
    });

    // Після завантаження плагінів
    $this->registerAction('plugins_loaded', function() {
        // Перевірка залежностей
        $this->checkDependencies();
    });
}
```

## Пріоритети хуків

Пріоритет визначає порядок виконання. Нижче пріоритет = вище пріоритет виконання.

**Стандартні пріоритети:**
- `1-5` — дуже високий
- `6-9` — високий
- `10` — стандартний (за замовчуванням)
- `11-19` — низький
- `20+` — дуже низький

**Приклад:**

```php
// Виконається першим
$this->registerAction('init', [$this, 'earlyInit'], 5);

// Виконається другим
$this->registerAction('init', [$this, 'normalInit'], 10);

// Виконається третім
$this->registerAction('init', [$this, 'lateInit'], 20);
```

## Ізоляція хуків

Хуки плагінів автоматично ізольовані через `PluginContainer`:

- Хуки реєструються з ізоляцією
- Конфлікти між плагінами уникнені
- Кожен плагін має власний контекст

## Найкращі практики

### 1. Використовуйте Filters для зміни даних

```php
// Добре: використання filter
$this->registerFilter('the_title', function($title) {
    return strtoupper($title);
});

// Погано: пряма зміна
$title = strtoupper($title); // Не дозволяє іншим плагінам змінювати
```

### 2. Використовуйте Actions для виконання дій

```php
// Добре: використання action
$this->registerAction('init', function() {
    // Виконання дії
});

// Погано: виконання без action
// Код виконується завжди, навіть якщо плагін деактивований
```

### 3. Перевіряйте наявність хуків

```php
// Перевірка перед виконанням
if (Action::has('custom_action')) {
    Action::do('custom_action');
}
```

### 4. Документуйте власні хуки

```php
/**
 * Action: my_plugin_loaded
 * Виконується після завантаження плагіна
 *
 * @param string $pluginSlug Slug плагіна
 */
$this->registerAction('my_plugin_loaded', function($pluginSlug) {
    // ...
});
```

## Наступні кроки

- [Створення плагіна](Creating-Plugin) — покрокове створення
- [Життєвий цикл](Plugin-Lifecycle) — детальний опис життєвого циклу
- [Адмін-панель](Plugin-Admin) — створення адмін-інтерфейсу

---

**Хуки забезпечують гнучку інтеграцію плагінів з системою!** 🎣
