# Адмін-панель для плагінів

Детальний опис створення адмін-інтерфейсу для плагінів.

## Огляд

Плагіни можуть створювати власні сторінки в адмін-панелі для управління налаштуваннями та функціональністю.

## Створення адмін-сторінки

### Структура

Створіть файл `admin/pages/my-plugin.php`:

```php
<?php

/**
 * Сторінка адмінки плагіна
 */

function admin_my_plugin_page(): void
{
    // Обробка POST запитів
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Логіка обробки
    }

    // Рендеринг сторінки
    render_admin_layout('my-plugin', [
        'title' => 'Мій плагін',
    ]);
}
```

### Додавання в меню

Реєстрація пункту меню через filter:

```php
public function registerHooks(): void
{
    $this->registerFilter('admin_menu', function (array $menu): array {
        $menu[] = [
            'text' => 'Мій плагін',
            'title' => 'Управління моїм плагіном',
            'icon' => 'fas fa-cog',
            'href' => admin_url('my-plugin'),
            'page' => 'my-plugin',
            'permission' => 'admin.access',
            'order' => 50,
        ];
        return $menu;
    });
}
```

### Реєстрація маршруту

Реєстрація маршруту через hook:

```php
public function registerRoutes(): void
{
    $this->registerHook('admin_routes', function ($router) {
        $pageFile = __DIR__ . DS . 'admin' . DS . 'pages' . DS . 'my-plugin.php';
        if (file_exists($pageFile)) {
            require_once $pageFile;
        }

        if ($router && method_exists($router, 'add')) {
            $router->add(['GET', 'POST'], 'my-plugin', 'admin_my_plugin_page', [
                'middleware' => ['admin.auth']
            ]);
        }
    });
}
```

## Обробка форм

### Збереження даних

```php
function admin_my_plugin_page(): void
{
    $plugin = new Plugin();

    // Обробка POST
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['action']) && $_POST['action'] === 'save') {
            // Валідація CSRF
            if (!verify_csrf_token($_POST['_token'] ?? '')) {
                set_flash_message('Помилка безпеки', 'error');
                admin_redirect('/admin/my-plugin');
                exit;
            }

            // Збереження налаштувань
            $plugin->setSetting('setting1', $_POST['setting1'] ?? '');
            $plugin->setSetting('setting2', $_POST['setting2'] ?? '');

            set_flash_message('Налаштування збережено', 'success');
            admin_redirect('/admin/my-plugin');
            exit;
        }
    }

    // Отримання налаштувань
    $settings = $plugin->getSettings();

    // Рендеринг
    render_admin_layout('my-plugin', [
        'title' => 'Мій плагін',
        'settings' => $settings,
    ]);
}
```

### Видалення даних

```php
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'delete') {
        // Валідація CSRF
        if (!verify_csrf_token($_POST['_token'] ?? '')) {
            set_flash_message('Помилка безпеки', 'error');
            admin_redirect('/admin/my-plugin');
            exit;
        }

        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            // Видалення
            delete_item($id);
            set_flash_message('Елемент видалено', 'success');
        }

        admin_redirect('/admin/my-plugin');
        exit;
    }
}
```

## Використання компонентів адмінки

### Кнопки

```php
// У шаблоні
<?php
$buttons = [
    [
        'text' => 'Зберегти',
        'type' => 'submit',
        'class' => 'btn-primary',
    ],
    [
        'text' => 'Скасувати',
        'href' => admin_url('my-plugin'),
        'class' => 'btn-secondary',
    ],
];
render_admin_buttons($buttons);
?>
```

### Форми

```php
// У шаблоні
<?php
$form = [
    'action' => admin_url('my-plugin'),
    'method' => 'POST',
    'fields' => [
        [
            'type' => 'text',
            'name' => 'setting1',
            'label' => 'Налаштування 1',
            'value' => $settings['setting1'] ?? '',
            'required' => true,
        ],
        [
            'type' => 'textarea',
            'name' => 'setting2',
            'label' => 'Налаштування 2',
            'value' => $settings['setting2'] ?? '',
        ],
    ],
];
render_admin_form($form);
?>
```

### Таблиці

```php
// У шаблоні
<?php
$table = [
    'headers' => ['ID', 'Назва', 'Дії'],
    'rows' => [
        [1, 'Елемент 1', '<a href="...">Редагувати</a>'],
        [2, 'Елемент 2', '<a href="...">Редагувати</a>'],
    ],
];
render_admin_table($table);
?>
```

### Модальні вікна

```php
// У шаблоні
<?php
$modal = [
    'id' => 'my-modal',
    'title' => 'Підтвердження',
    'content' => 'Ви впевнені?',
    'buttons' => [
        [
            'text' => 'Так',
            'class' => 'btn-primary',
            'onclick' => 'confirmAction()',
        ],
        [
            'text' => 'Ні',
            'class' => 'btn-secondary',
            'data-dismiss' => 'modal',
        ],
    ],
];
render_admin_modal($modal);
?>
```

## Flash повідомлення

### Встановлення повідомлення

```php
set_flash_message('Налаштування збережено', 'success');
set_flash_message('Помилка збереження', 'error');
set_flash_message('Попередження', 'warning');
set_flash_message('Інформація', 'info');
```

### Відображення повідомлення

Повідомлення автоматично відображаються в адмін-панелі через `render_admin_layout()`.

## Редиректи

### Після збереження

```php
// POST-Redirect-GET паттерн
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Збереження
    save_data();

    set_flash_message('Дані збережено', 'success');
    admin_redirect('/admin/my-plugin');
    exit;
}
```

### З параметрами

```php
admin_redirect('/admin/my-plugin?tab=settings');
```

## Приклад повної сторінки

```php
<?php

function admin_my_plugin_page(): void
{
    $plugin = new Plugin();

    // Обробка POST
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['action'])) {
            // CSRF захист
            if (!verify_csrf_token($_POST['_token'] ?? '')) {
                set_flash_message('Помилка безпеки', 'error');
                admin_redirect('/admin/my-plugin');
                exit;
            }

            switch ($_POST['action']) {
                case 'save':
                    $plugin->setSetting('setting1', $_POST['setting1'] ?? '');
                    $plugin->setSetting('setting2', $_POST['setting2'] ?? '');
                    set_flash_message('Налаштування збережено', 'success');
                    break;

                case 'delete':
                    $id = (int)($_POST['id'] ?? 0);
                    if ($id > 0) {
                        delete_item($id);
                        set_flash_message('Елемент видалено', 'success');
                    }
                    break;
            }

            admin_redirect('/admin/my-plugin');
            exit;
        }
    }

    // Отримання даних
    $settings = $plugin->getSettings();
    $items = get_items();

    // Рендеринг
    render_admin_layout('my-plugin', [
        'title' => 'Мій плагін',
        'settings' => $settings,
        'items' => $items,
    ]);
}
```

## Створення шаблону

Створіть файл `admin/templates/my-plugin.php`:

```php
<?php
/**
 * @var string $title
 * @var array $settings
 * @var array $items
 */
?>

<div class="card">
    <div class="card-header">
        <h3><?php echo htmlspecialchars($title); ?></h3>
    </div>
    <div class="card-body">
        <form method="POST" action="<?php echo admin_url('my-plugin'); ?>">
            <?php echo csrf_token_field(); ?>
            <input type="hidden" name="action" value="save">

            <div class="form-group">
                <label>Налаштування 1</label>
                <input type="text" name="setting1"
                       value="<?php echo htmlspecialchars($settings['setting1'] ?? ''); ?>"
                       class="form-control">
            </div>

            <div class="form-group">
                <label>Налаштування 2</label>
                <textarea name="setting2" class="form-control"><?php
                    echo htmlspecialchars($settings['setting2'] ?? '');
                ?></textarea>
            </div>

            <button type="submit" class="btn btn-primary">Зберегти</button>
        </form>
    </div>
</div>
```

## Найкращі практики

### 1. Використовуйте POST-Redirect-GET

```php
// Добре: редирект після POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    save_data();
    admin_redirect('/admin/my-plugin');
    exit;
}

// Погано: рендеринг після POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    save_data();
    // Рендеринг без редиректу
}
```

### 2. Валідуйте CSRF токени

```php
// Завжди перевіряйте CSRF
if (!verify_csrf_token($_POST['_token'] ?? '')) {
    set_flash_message('Помилка безпеки', 'error');
    admin_redirect('/admin/my-plugin');
    exit;
}
```

### 3. Валідуйте дані

```php
// Валідація вхідних даних
$setting1 = sanitize_input($_POST['setting1'] ?? '');
if (empty($setting1)) {
    set_flash_message('Поле обов\'язкове', 'error');
    admin_redirect('/admin/my-plugin');
    exit;
}
```

### 4. Використовуйте flash повідомлення

```php
// Завжди повідомляйте про результат
set_flash_message('Дані збережено', 'success');
// або
set_flash_message('Помилка збереження', 'error');
```

## Наступні кроки

- [Створення плагіна](Creating-Plugin) — покрокове створення
- [Життєвий цикл](Plugin-Lifecycle) — детальний опис життєвого циклу
- [Хуки для плагінів](Plugin-Hooks) — використання хуків

---

**Адмін-панель забезпечує зручне управління плагіном!** 🎛️
