# Огляд системи плагінів

Загальний огляд системи плагінів FLOWAXY-CMS та їх архітектури.

## Що таке плагін?

Плагін — це модульне розширення функціональності системи, яке може бути встановлено, активовано, деактивовано та видалено без зміни ядра.

## Переваги системи плагінів

- **Модульність** — кожен плагін незалежний
- **Ізоляція** — плагіни не впливають один на одного
- **Гнучкість** — легко додавати та видаляти функціональність
- **Розширюваність** — інтеграція через хуки та фільтри
- **Безпека** — ізоляція коду та даних

## Архітектура плагіна

### Базовий клас

Всі плагіни успадковують `BasePlugin`:

```php
<?php
namespace Flowaxy\Plugins\MyPlugin;

use Flowaxy\Support\Base\BasePlugin;

final class Plugin extends BasePlugin
{
    public function registerHooks(): void
    {
        // Реєстрація хуків
    }

    public function registerRoutes(): void
    {
        // Реєстрація маршрутів
    }
}
```

### Структура плагіна

```
content/plugins/[plugin-name]/
├── Plugin.php           # Клас плагіна (обов'язково)
├── admin/               # Адмін-інтерфейс
│   └── pages/           # Сторінки адмінки
│       └── [page].php
├── config/              # Конфігурація
│   ├── plugin.ini       # Метадані (опціонально)
│   └── [config].ini     # Конфігураційні файли
└── README.md            # Документація
```

## Життєвий цикл плагіна

### Стадії життєвого циклу

1. **Install** — встановлення плагіна
2. **Activate** — активація плагіна
3. **Deactivate** — деактивація плагіна
4. **Uninstall** — видалення плагіна

### Методи життєвого циклу

```php
class Plugin extends BasePlugin
{
    // Викликається при встановленні (один раз)
    public function install(): void
    {
        // Створення таблиць БД, налаштувань
    }

    // Викликається при активації
    public function activate(): void
    {
        // Ініціалізація, реєстрація хуків
    }

    // Викликається при деактивації
    public function deactivate(): void
    {
        // Очищення тимчасових даних
    }

    // Викликається при видаленні
    public function uninstall(): void
    {
        // Видалення таблиць БД, налаштувань
    }
}
```

**Детальніше:** [Життєвий цикл плагіна](Plugin-Lifecycle)

## Реєстрація хуків

Плагіни реєструють хуки для інтеграції з системою:

```php
public function registerHooks(): void
{
    // Action
    $this->registerAction('init', [$this, 'onInit']);

    // Filter
    $this->registerFilter('the_title', [$this, 'formatTitle']);
}
```

**Детальніше:** [Хуки для плагінів](Plugin-Hooks)

## Реєстрація маршрутів

Плагіни можуть реєструвати власні маршрути:

```php
public function registerRoutes(): void
{
    $this->registerHook('admin_routes', function ($router) {
        $router->add(['GET', 'POST'], 'my-plugin', 'admin_my_plugin_page');
    });
}
```

## Адмін-панель

Плагіни можуть створювати сторінки в адмін-панелі:

```php
public function registerHooks(): void
{
    $this->registerFilter('admin_menu', function (array $menu): array {
        $menu[] = [
            'text' => 'Мій плагін',
            'href' => admin_url('my-plugin'),
            'page' => 'my-plugin',
        ];
        return $menu;
    });
}
```

**Детальніше:** [Адмін-панель для плагінів](Plugin-Admin)

## Налаштування плагіна

Плагіни можуть зберігати налаштування:

```php
// Отримання налаштувань
$settings = $this->getSettings();

// Збереження налаштування
$this->setSetting('key', 'value');
```

Налаштування зберігаються в `storage/config/plugins/{plugin-slug}.ini`.

## Ізоляція плагінів

Плагіни ізольовані один від одного:

- **Окремий контейнер** — кожен плагін має власний контейнер
- **Ізольовані хуки** — хуки реєструються з ізоляцією
- **Окремі налаштування** — кожен плагін має власні налаштування

## Приклади плагінів

### Social Networks Plugin

Повнофункціональний приклад плагіна для управління соціальними мережами.

**Детальніше:** [Приклади плагінів](Examples)

## Наступні кроки

- [Створення плагіна](Creating-Plugin) — покрокове створення
- [Життєвий цикл](Plugin-Lifecycle) — детальний опис життєвого циклу
- [Хуки для плагінів](Plugin-Hooks) — використання хуків
- [Адмін-панель](Plugin-Admin) — створення адмін-інтерфейсу
- [Приклади](Examples) — готові приклади плагінів

---

**Система плагінів забезпечує гнучкість та розширюваність!** 🔌
