# Приклади плагінів

Повні приклади плагінів для FLOWAXY-CMS з детальним описом.

## Social Networks Plugin

Повнофункціональний плагін для управління соціальними мережами.

### Опис

Плагін дозволяє управляти списком соціальних мереж через адмін-панель та надає дані через хуки для відображення в темі.

### Структура

```
content/plugins/social-networks/
├── Plugin.php
├── admin/
│   └── pages/
│       └── social-networks.php
├── config/
│   └── networks.ini
└── README.md
```

### Plugin.php

```php
<?php

/**
 * Social Networks Plugin
 *
 * @name: Social Networks
 * @slug: social-networks
 * @version: 1.0.0
 * @description: Плагін для управління соціальними мережами
 * @author: Flowaxy Team
 * @author_url: https://flowaxy.com
 * @requires: 1.0.0
 * @tested: 1.0.0
 * @package: SocialNetworks
 */

declare(strict_types=1);

use Flowaxy\Support\Base\BasePlugin;

final class SocialNetworksPlugin extends BasePlugin
{
    public function registerHooks(): void
    {
        // Додавання в меню адмінки
        $this->registerFilter('admin_menu', function (array $menu): array {
            $adminUrl = function_exists('admin_url')
                ? admin_url('social-networks')
                : '/admin/social-networks';
            $menu[] = [
                'text' => 'Соціальні мережі',
                'title' => 'Управління соціальними мережами',
                'icon' => 'fas fa-share-alt',
                'href' => $adminUrl,
                'page' => 'social-networks',
                'permission' => 'admin.access',
                'order' => 46,
            ];
            return $menu;
        });

        // Надання даних для теми
        $this->registerFilter('theme_social_networks', function (array $networks): array {
            $settings = $this->getSettings();
            if (($settings['status'] ?? 'activated') === 'deactivated') {
                return $networks;
            }
            return array_merge($networks, $this->getSocialNetworks());
        });
    }

    public function registerRoutes(): void
    {
        $this->registerHook('admin_routes', function ($router) {
            $pageFile = __DIR__ . DS . 'admin' . DS . 'pages' . DS . 'social-networks.php';
            if (file_exists($pageFile) && is_readable($pageFile)) {
                require_once $pageFile;
            }
            if ($router && method_exists($router, 'add')) {
                $router->add(['GET', 'POST'], 'social-networks', 'admin_social_networks_page', [
                    'middleware' => ['admin.auth']
                ]);
            }
        });
    }

    private function getSocialNetworks(): array
    {
        $configFile = __DIR__ . DS . 'config' . DS . 'networks.ini';
        if (!file_exists($configFile)) {
            return [];
        }

        $networks = [];
        try {
            $config = parse_ini_file($configFile, true);
            if (!is_array($config)) {
                return [];
            }

            foreach ($config as $sectionName => $sectionData) {
                if (!is_array($sectionData) || strpos($sectionName, 'networks.') !== 0) {
                    continue;
                }

                $id = substr($sectionName, strlen('networks.'));
                $enabled = isset($sectionData['enabled']) ? (bool)(int)$sectionData['enabled'] : true;
                if (!$enabled) {
                    continue;
                }

                $networks[] = [
                    'id' => (string)$id,
                    'name' => $sectionData['name'] ?? $id,
                    'url' => $sectionData['url'] ?? '#',
                    'icon' => $sectionData['icon'] ?? 'fas fa-link',
                    'order' => (int)($sectionData['order'] ?? 999),
                    'enabled' => $enabled,
                ];
            }
        } catch (\Throwable $e) {
            return [];
        }

        usort($networks, fn($a, $b) => $a['order'] <=> $b['order']);
        return $networks;
    }

    public function activate(): void
    {
        parent::activate();
        $this->initializeDefaultNetworks();
    }

    private function initializeDefaultNetworks(): void
    {
        $configFile = __DIR__ . DS . 'config' . DS . 'networks.ini';
        if (file_exists($configFile)) {
            return;
        }

        $defaultNetworks = [
            'vk' => ['name' => 'VKontakte', 'url' => 'https://vk.com', 'icon' => 'fab fa-vk', 'order' => 1],
            'twitter' => ['name' => 'Twitter/X', 'url' => 'https://twitter.com', 'icon' => 'fab fa-twitter', 'order' => 2],
            'facebook' => ['name' => 'Facebook', 'url' => 'https://facebook.com', 'icon' => 'fab fa-facebook', 'order' => 3],
        ];

        $iniContent = "; Social Networks Configuration\n\n[networks]\n\n";
        foreach ($defaultNetworks as $id => $network) {
            $iniContent .= "[networks.{$id}]\n";
            $iniContent .= "name = \"{$network['name']}\"\n";
            $iniContent .= "url = \"{$network['url']}\"\n";
            $iniContent .= "icon = \"{$network['icon']}\"\n";
            $iniContent .= "order = {$network['order']}\n";
            $iniContent .= "enabled = 1\n\n";
        }

        $configDir = dirname($configFile);
        if (!is_dir($configDir)) {
            mkdir($configDir, 0755, true);
        }
        file_put_contents($configFile, $iniContent);
    }
}
```

### Використання в темі

```php
<?php
// У шаблоні теми
$networks = [];
if (class_exists('\Flowaxy\Core\Hooks\Filter')) {
    $networks = \Flowaxy\Core\Hooks\Filter::apply('theme_social_networks', $networks);
}

if (!empty($networks)) {
    echo '<div class="social-share">';
    foreach ($networks as $network) {
        if (!($network['enabled'] ?? true) || empty($network['url'])) {
            continue;
        }
        echo '<a href="' . htmlspecialchars($network['url']) . '"
                  class="share-icon"
                  target="_blank">
                  <i class="' . htmlspecialchars($network['icon']) . '"></i>
              </a>';
    }
    echo '</div>';
}
?>
```

## Простий приклад плагіна

Мінімальний приклад плагіна з базовою функціональністю.

### Plugin.php

```php
<?php

/**
 * Simple Plugin
 *
 * @name: Simple Plugin
 * @slug: simple-plugin
 * @version: 1.0.0
 * @description: Простий приклад плагіна
 * @author: Your Name
 * @requires: 1.0.0
 * @tested: 1.0.0
 * @package: SimplePlugin
 */

declare(strict_types=1);

use Flowaxy\Support\Base\BasePlugin;

final class Plugin extends BasePlugin
{
    public function registerHooks(): void
    {
        // Додавання в меню
        $this->registerFilter('admin_menu', function (array $menu): array {
            $menu[] = [
                'text' => 'Simple Plugin',
                'href' => admin_url('simple-plugin'),
                'page' => 'simple-plugin',
            ];
            return $menu;
        });

        // Фільтрація заголовка
        $this->registerFilter('the_title', function(string $title): string {
            return '[Plugin] ' . $title;
        });
    }

    public function registerRoutes(): void
    {
        $this->registerHook('admin_routes', function ($router) {
            $pageFile = __DIR__ . DS . 'admin' . DS . 'pages' . DS . 'simple-plugin.php';
            if (file_exists($pageFile)) {
                require_once $pageFile;
            }
            if ($router && method_exists($router, 'add')) {
                $router->add(['GET', 'POST'], 'simple-plugin', 'admin_simple_plugin_page');
            }
        });
    }
}
```

### admin/pages/simple-plugin.php

```php
<?php

function admin_simple_plugin_page(): void
{
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['action']) && $_POST['action'] === 'save') {
            if (!verify_csrf_token($_POST['_token'] ?? '')) {
                set_flash_message('Помилка безпеки', 'error');
                admin_redirect('/admin/simple-plugin');
                exit;
            }

            $plugin = new Plugin();
            $plugin->setSetting('message', $_POST['message'] ?? '');

            set_flash_message('Налаштування збережено', 'success');
            admin_redirect('/admin/simple-plugin');
            exit;
        }
    }

    $plugin = new Plugin();
    $settings = $plugin->getSettings();

    render_admin_layout('simple-plugin', [
        'title' => 'Simple Plugin',
        'settings' => $settings,
    ]);
}
```

## Наступні кроки

- [Створення плагіна](Creating-Plugin) — покрокове створення
- [Життєвий цикл](Plugin-Lifecycle) — детальний опис життєвого циклу
- [Хуки для плагінів](Plugin-Hooks) — використання хуків
- [Адмін-панель](Plugin-Admin) — створення адмін-інтерфейсу

---

**Приклади плагінів допоможуть швидко розпочати розробку!** 📚
