# Створення плагіна

Покрокове керівництво з створення плагіна для FLOWAXY-CMS.

## Крок 1: Створення структури

### Створення директорії

Створіть директорію для плагіна в `content/plugins/`:

```
content/plugins/my-plugin/
```

### Базова структура

```
content/plugins/my-plugin/
├── Plugin.php           # Клас плагіна (обов'язково)
├── admin/               # Адмін-інтерфейс (опціонально)
│   └── pages/           # Сторінки адмінки
├── config/              # Конфігурація (опціонально)
└── README.md            # Документація (рекомендовано)
```

## Крок 2: Створення класу плагіна

### Базовий клас

Створіть файл `Plugin.php`:

```php
<?php

/**
 * My Plugin
 *
 * @name: My Plugin
 * @slug: my-plugin
 * @version: 1.0.0
 * @description: Опис плагіна
 * @author: Ваше ім'я
 * @author_url: https://example.com
 * @requires: 1.0.0
 * @tested: 1.0.0
 * @package: MyPlugin
 *
 * FLOWAXY CMS 1.0.0 Stable
 */

declare(strict_types=1);

use Flowaxy\Support\Base\BasePlugin;

final class Plugin extends BasePlugin
{
    public function registerHooks(): void
    {
        // Реєстрація хуків
    }

    public function registerRoutes(): void
    {
        // Реєстрація маршрутів
    }
}
```

### Метадані плагіна

Метадані вказуються в PHPDoc блоці на початку файлу:

- `@name` — назва плагіна
- `@slug` — унікальний ідентифікатор (використовується в URL та конфігах)
- `@version` — версія плагіна
- `@description` — опис функціональності
- `@author` — автор плагіна
- `@author_url` — URL автора
- `@requires` — мінімальна версія FLOWAXY-CMS
- `@tested` — версія, на якій протестовано
- `@package` — namespace пакету

## Крок 3: Реєстрація хуків

### Реєстрація Actions

```php
public function registerHooks(): void
{
    // Реєстрація action
    $this->registerAction('init', [$this, 'onInit']);

    // Або через замикання
    $this->registerAction('admin_init', function() {
        // Логіка ініціалізації адмінки
    });
}
```

### Реєстрація Filters

```php
public function registerHooks(): void
{
    // Реєстрація filter
    $this->registerFilter('the_title', [$this, 'formatTitle']);

    // Або через замикання
    $this->registerFilter('admin_menu', function(array $menu): array {
        $menu[] = [
            'text' => 'Мій плагін',
            'href' => admin_url('my-plugin'),
        ];
        return $menu;
    });
}
```

## Крок 4: Реєстрація маршрутів

### Маршрути адмінки

```php
public function registerRoutes(): void
{
    $this->registerHook('admin_routes', function ($router) {
        $pageFile = __DIR__ . DS . 'admin' . DS . 'pages' . DS . 'my-plugin.php';
        if (file_exists($pageFile)) {
            require_once $pageFile;
        }

        if ($router && method_exists($router, 'add')) {
            $router->add(['GET', 'POST'], 'my-plugin', 'admin_my_plugin_page', [
                'middleware' => ['admin.auth']
            ]);
        }
    });
}
```

## Крок 5: Створення адмін-сторінки

### Структура сторінки

Створіть файл `admin/pages/my-plugin.php`:

```php
<?php

/**
 * Сторінка адмінки плагіна
 */

function admin_my_plugin_page(): void
{
    // Обробка POST запитів
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Збереження даних
        if (isset($_POST['action']) && $_POST['action'] === 'save') {
            // Логіка збереження
        }

        // Редирект після збереження
        admin_redirect('/admin/my-plugin');
        exit;
    }

    // Рендеринг сторінки
    render_admin_layout('my-plugin', [
        'title' => 'Мій плагін',
        'data' => [],
    ]);
}
```

### Додавання в меню

```php
public function registerHooks(): void
{
    $this->registerFilter('admin_menu', function (array $menu): array {
        $menu[] = [
            'text' => 'Мій плагін',
            'title' => 'Управління моїм плагіном',
            'icon' => 'fas fa-cog',
            'href' => admin_url('my-plugin'),
            'page' => 'my-plugin',
            'permission' => 'admin.access',
            'order' => 50,
        ];
        return $menu;
    });
}
```

## Крок 6: Налаштування плагіна

### Отримання налаштувань

```php
// Отримання всіх налаштувань
$settings = $this->getSettings();

// Отримання конкретного налаштування
$value = $settings['key'] ?? 'default';
```

### Збереження налаштувань

```php
// Збереження одного налаштування
$this->setSetting('key', 'value');

// Збереження кількох налаштувань
foreach ($data as $key => $value) {
    $this->setSetting($key, $value);
}
```

Налаштування автоматично зберігаються в `storage/config/plugins/{plugin-slug}.ini`.

## Крок 7: Життєвий цикл

### Метод install()

```php
public function install(): void
{
    parent::install();

    // Створення таблиць БД
    // Ініціалізація конфігурацій
    // Створення директорій
}
```

### Метод activate()

```php
public function activate(): void
{
    parent::activate();

    // Ініціалізація даних
    // Реєстрація хуків
    // Створення тимчасових файлів
}
```

### Метод deactivate()

```php
public function deactivate(): void
{
    parent::deactivate();

    // Очищення тимчасових даних
    // Зупинка фонових процесів
}
```

### Метод uninstall()

```php
public function uninstall(): void
{
    parent::uninstall();

    // Видалення таблиць БД
    // Видалення конфігурацій
    // Видалення файлів
}
```

## Крок 8: Тестування

### Перевірка встановлення

1. Перейдіть в адмін-панель → **Плагіни**
2. Знайдіть ваш плагін в списку
3. Натисніть **Встановити**

### Перевірка активації

1. Після встановлення натисніть **Активувати**
2. Перевірте, що плагін з'явився в меню (якщо додано)
3. Перевірте роботу хуків

### Перевірка функціональності

1. Відкрийте сторінку плагіна в адмінці
2. Перевірте збереження налаштувань
3. Перевірте роботу всіх функцій

## Повний приклад

```php
<?php

/**
 * Example Plugin
 *
 * @name: Example Plugin
 * @slug: example-plugin
 * @version: 1.0.0
 * @description: Приклад плагіна
 * @author: Flowaxy Team
 * @author_url: https://flowaxy.com
 * @requires: 1.0.0
 * @tested: 1.0.0
 * @package: ExamplePlugin
 */

declare(strict_types=1);

use Flowaxy\Support\Base\BasePlugin;

final class Plugin extends BasePlugin
{
    public function registerHooks(): void
    {
        // Додавання в меню адмінки
        $this->registerFilter('admin_menu', function (array $menu): array {
            $menu[] = [
                'text' => 'Example Plugin',
                'href' => admin_url('example-plugin'),
                'page' => 'example-plugin',
            ];
            return $menu;
        });

        // Фільтрація заголовка
        $this->registerFilter('the_title', function(string $title): string {
            return strtoupper($title);
        });
    }

    public function registerRoutes(): void
    {
        $this->registerHook('admin_routes', function ($router) {
            $pageFile = __DIR__ . DS . 'admin' . DS . 'pages' . DS . 'example-plugin.php';
            if (file_exists($pageFile)) {
                require_once $pageFile;
            }

            if ($router && method_exists($router, 'add')) {
                $router->add(['GET', 'POST'], 'example-plugin', 'admin_example_plugin_page');
            }
        });
    }

    public function activate(): void
    {
        parent::activate();
        // Ініціалізація при активації
    }
}
```

## Наступні кроки

- [Життєвий цикл](Plugin-Lifecycle) — детальний опис життєвого циклу
- [Хуки для плагінів](Plugin-Hooks) — використання хуків
- [Адмін-панель](Plugin-Admin) — створення адмін-інтерфейсу
- [Приклади](Examples) — готові приклади плагінів

---

**Плагін створено! Тепер можна розширювати функціональність!** 🎉
