# Огляд адмін-панелі

Загальний огляд адмін-панелі FLOWAXY-CMS та її архітектури.

## Що таке адмін-панель?

Адмін-панель — це інтерфейс для управління системою, плагінами, темами та налаштуваннями.

## Архітектура адмін-панелі

### Структура

```
content/admin/
├── components/        # Компоненти конструктора (масиви)
│   ├── Button/       # Кнопки
│   ├── Form/         # Форми
│   ├── Modal/        # Модальні вікна
│   ├── Notification/  # Сповіщення
│   ├── Table/        # Таблиці
│   ├── Layout/       # Layout компоненти
│   └── Menu/         # Меню
├── pages/             # Сторінки адмінки (функції)
├── layouts/           # Макети (main.php, login.php)
├── partials/          # Частини (header, sidebar, footer)
├── includes/          # Допоміжні файли
│   ├── admin-routes.php
│   ├── functions.php
│   └── components.php
└── resources/         # Ресурси
    ├── styles/
    ├── scripts/
    └── images/
```

## Принципи роботи

### Функції замість класів

Всі сторінки адмінки — це функції без класів:

```php
function admin_dashboard_page(): void
{
    render_admin_layout([
        'title' => 'Панель управління',
        'content' => 'Контент сторінки',
    ]);
}
```

### Компоненти через масиви

Компоненти визначаються через масиви конфігурації:

```php
$button = [
    'text' => 'Зберегти',
    'type' => 'submit',
    'class' => 'btn-primary',
];
render_button($button);
```

### Автоматична авторизація

Авторизація обробляється автоматично через middleware `admin.auth`:

```php
// Автоматично застосовується до всіх маршрутів крім login/logout
$router->add(['GET', 'POST'], 'dashboard', 'admin_dashboard_page', [
    'middleware' => ['admin.auth']
]);
```

## Компоненти адмінки

### Button

Кнопки для дій.

**Використання:**
```php
render_button([
    'text' => 'Зберегти',
    'type' => 'submit',
    'class' => 'btn-primary',
]);
```

### Form

Форми для введення даних.

**Використання:**
```php
render_form([
    'action' => admin_url('settings'),
    'method' => 'POST',
    'fields' => [
        ['type' => 'text', 'name' => 'setting1', 'label' => 'Налаштування 1'],
    ],
]);
```

### Modal

Модальні вікна для діалогів.

**Використання:**
```php
render_modal([
    'id' => 'my-modal',
    'title' => 'Підтвердження',
    'content' => 'Ви впевнені?',
]);
```

### Notification

Сповіщення для повідомлень користувачу.

**Використання:**
```php
render_notification([
    'type' => 'success',
    'message' => 'Дані збережено',
]);
```

### Table

Таблиці для відображення даних.

**Використання:**
```php
render_table([
    'headers' => ['ID', 'Назва', 'Дії'],
    'rows' => [
        [1, 'Елемент 1', '<a href="...">Редагувати</a>'],
    ],
]);
```

**Детальніше:** [Компоненти](Components)

## Створення сторінок

### Базова структура

```php
function admin_my_page(): void
{
    // Обробка POST
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Логіка обробки
    }

    // Рендеринг
    render_admin_layout([
        'title' => 'Моя сторінка',
        'content' => 'Контент',
    ]);
}
```

### Breadcrumbs

```php
render_admin_layout([
    'title' => 'Моя сторінка',
    'pageBreadcrumbs' => [
        ['text' => 'Головна', 'href' => admin_url('dashboard')],
        ['text' => 'Моя сторінка'],
    ],
    'content' => 'Контент',
]);
```

### Header Buttons

```php
render_admin_layout([
    'title' => 'Моя сторінка',
    'pageHeaderButtons' => [
        [
            'text' => 'Додати',
            'href' => admin_url('my-page/add'),
            'class' => 'btn-primary',
        ],
    ],
    'content' => 'Контент',
]);
```

**Детальніше:** [Створення сторінок](Creating-Pages)

## Реєстрація маршрутів

### Автоматична реєстрація

Маршрути автоматично завантажуються з `content/admin/includes/admin-routes.php`:

```php
$router->add(['GET', 'POST'], 'my-page', 'admin_my_page', [
    'middleware' => ['admin.auth']
]);
```

### Реєстрація з плагінів

Плагіни можуть реєструвати власні маршрути:

```php
$this->registerHook('admin_routes', function ($router) {
    $router->add(['GET', 'POST'], 'my-plugin', 'admin_my_plugin_page');
});
```

## Меню адмінки

### Реєстрація пунктів меню

```php
add_filter('admin_menu', function (array $menu): array {
    $menu[] = [
        'text' => 'Мій розділ',
        'href' => admin_url('my-page'),
        'page' => 'my-page',
        'icon' => 'fas fa-cog',
        'order' => 50,
    ];
    return $menu;
});
```

## Найкращі практики

### 1. POST-Redirect-GET

```php
// Добре: Редирект після POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    save_data();
    admin_redirect('my-page');
    exit;
}

// Погано: Рендеринг після POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    save_data();
    // Рендеринг без редиректу
}
```

### 2. CSRF захист

```php
// Завжди перевіряйте CSRF
if (!verify_csrf_token($_POST['_token'] ?? '')) {
    set_flash_message('Помилка безпеки', 'error');
    admin_redirect('my-page');
    exit;
}
```

### 3. Flash повідомлення

```php
// Завжди повідомляйте про результат
set_flash_message('Дані збережено', 'success');
// або
set_flash_message('Помилка збереження', 'error');
```

## Наступні кроки

- [Створення сторінок](Creating-Pages) — покрокове створення сторінок
- [Компоненти](Components) — використання компонентів
- [Кастомізація](Customization) — кастомізація адмін-панелі

---

**Адмін-панель забезпечує зручне управління системою!** 🎛️
