# Кастомізація адмін-панелі

Детальний опис кастомізації адмін-панелі та додавання власних компонентів.

## Огляд

Адмін-панель можна кастомізувати через:
- Власні компоненти
- Власні стилі
- Власні скрипти
- Модифікацію layout

## Кастомізація меню

### Додавання пунктів меню

```php
// У плагіні або темі
add_filter('admin_menu', function (array $menu): array {
    $menu[] = [
        'text' => 'Мій розділ',
        'title' => 'Управління моїм розділом',
        'icon' => 'fas fa-cog',
        'href' => admin_url('my-section'),
        'page' => 'my-section',
        'permission' => 'admin.access',
        'order' => 50,
    ];
    return $menu;
});
```

### Підменю

```php
add_filter('admin_menu', function (array $menu): array {
    $menu[] = [
        'text' => 'Мій розділ',
        'icon' => 'fas fa-cog',
        'href' => '#',
        'submenu' => [
            [
                'text' => 'Підрозділ 1',
                'href' => admin_url('my-section/sub1'),
            ],
            [
                'text' => 'Підрозділ 2',
                'href' => admin_url('my-section/sub2'),
            ],
        ],
    ];
    return $menu;
});
```

## Кастомізація layout

### Модифікація header

Створіть власний header в `content/admin/partials/header.php` або модифікуйте існуючий.

### Модифікація sidebar

Створіть власний sidebar в `content/admin/partials/sidebar.php` або модифікуйте існуючий.

### Модифікація footer

Створіть власний footer в `content/admin/partials/footer.php` або модифікуйте існуючий.

## Власні компоненти

### Створення компонента

Створіть директорію `content/admin/components/my-component/`:

```
components/my-component/
└── my-component.php
```

### Використання компонента

```php
includeComponent('my-component', [
    'param1' => 'value1',
    'param2' => 'value2',
]);
```

## Власні стилі

### Додавання CSS

Створіть файл `content/admin/resources/styles/custom.css`:

```css
.my-custom-class {
    background: #f0f0f0;
    padding: 20px;
}
```

### Підключення стилів

Додайте в `content/admin/layouts/main.php`:

```php
<link rel="stylesheet" href="/content/admin/resources/styles/custom.css">
```

## Власні скрипти

### Додавання JavaScript

Створіть файл `content/admin/resources/scripts/custom.js`:

```javascript
document.addEventListener('DOMContentLoaded', function() {
    console.log('Custom admin script loaded');
});
```

### Підключення скриптів

Додайте в `content/admin/layouts/main.php`:

```php
<script src="/content/admin/resources/scripts/custom.js"></script>
```

## Хуки для кастомізації

### admin_init

Виконується при ініціалізації адмінки.

```php
add_action('admin_init', function() {
    // Кастомізація при ініціалізації
});
```

### admin_menu

Фільтрація меню адмінки.

```php
add_filter('admin_menu', function(array $menu): array {
    // Модифікація меню
    return $menu;
});
```

### admin_footer

Додавання контенту в footer.

```php
add_action('admin_footer', function() {
    echo '<div>Custom footer content</div>';
});
```

## Найкращі практики

### 1. Не модифікуйте ядро

```php
// Добре: Використання хуків
add_filter('admin_menu', function($menu) {
    // Модифікація меню
    return $menu;
});

// Погано: Пряма модифікація файлів ядра
// Не редагуйте файли в content/admin/layouts/ безпосередньо
```

### 2. Використовуйте хуки

```php
// Добре: Хуки для розширення
add_action('admin_init', function() {
    // Логіка
});

// Погано: Пряме виконання коду
// Код виконується завжди, навіть якщо не потрібен
```

### 3. Ізоляція стилів

```css
/* Добре: Префікс для ізоляції */
.my-plugin-admin { }
.my-plugin-admin .item { }

/* Погано: Глобальні стилі */
.item { }
```

## Наступні кроки

- [Створення сторінок](Creating-Pages) — покрокове створення сторінок
- [Компоненти](Components) — використання компонентів

---

**Кастомізація забезпечує гнучкість та розширюваність!** 🎨
