# Створення сторінок адмінки

Покрокове керівництво з створення сторінок адмін-панелі.

## Крок 1: Створення функції сторінки

### Базова структура

Створіть файл `content/admin/pages/my-page.php`:

```php
<?php

declare(strict_types=1);

/**
 * Сторінка адмінки
 */
function admin_my_page(): void
{
    // Обробка POST запитів
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Логіка обробки
    }

    // Рендеринг сторінки
    render_admin_layout([
        'title' => 'Моя сторінка',
        'content' => 'Контент сторінки',
    ]);
}
```

## Крок 2: Обробка форм

### Збереження даних

```php
function admin_my_page(): void
{
    // Обробка POST
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['action']) && $_POST['action'] === 'save') {
            // CSRF захист
            if (!verify_csrf_token($_POST['_token'] ?? '')) {
                set_flash_message('Помилка безпеки', 'error');
                admin_redirect('my-page');
                exit;
            }

            // Валідація та збереження
            $setting1 = sanitize_input($_POST['setting1'] ?? '');
            if (empty($setting1)) {
                set_flash_message('Поле обов\'язкове', 'error');
                admin_redirect('my-page');
                exit;
            }

            // Збереження
            save_setting('setting1', $setting1);

            set_flash_message('Налаштування збережено', 'success');
            admin_redirect('my-page');
            exit;
        }
    }

    // Отримання даних
    $settings = get_settings();

    // Рендеринг
    render_admin_layout([
        'title' => 'Моя сторінка',
        'content' => render_my_page_content($settings),
    ]);
}
```

### Видалення даних

```php
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'delete') {
        // CSRF захист
        if (!verify_csrf_token($_POST['_token'] ?? '')) {
            set_flash_message('Помилка безпеки', 'error');
            admin_redirect('my-page');
            exit;
        }

        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            delete_item($id);
            set_flash_message('Елемент видалено', 'success');
        }

        admin_redirect('my-page');
        exit;
    }
}
```

## Крок 3: Breadcrumbs

### Додавання breadcrumbs

```php
render_admin_layout([
    'title' => 'Моя сторінка',
    'pageBreadcrumbs' => [
        ['text' => 'Головна', 'href' => admin_url('dashboard')],
        ['text' => 'Мій розділ', 'href' => admin_url('my-section')],
        ['text' => 'Моя сторінка'],
    ],
    'content' => 'Контент',
]);
```

### З іконкою

```php
render_admin_layout([
    'title' => 'Моя сторінка',
    'pageBreadcrumbIcon' => 'fas fa-cog',
    'pageBreadcrumbs' => [
        ['text' => 'Головна', 'href' => admin_url('dashboard')],
        ['text' => 'Моя сторінка'],
    ],
    'content' => 'Контент',
]);
```

## Крок 4: Header Buttons

### Додавання кнопок в header

```php
render_admin_layout([
    'title' => 'Моя сторінка',
    'pageHeaderButtons' => [
        [
            'text' => 'Додати',
            'href' => admin_url('my-page/add'),
            'class' => 'btn-primary',
            'icon' => 'fas fa-plus',
        ],
        [
            'text' => 'Налаштування',
            'href' => admin_url('my-page/settings'),
            'class' => 'btn-secondary',
            'icon' => 'fas fa-cog',
        ],
    ],
    'content' => 'Контент',
]);
```

## Крок 5: Використання компонентів

### Форма

```php
ob_start();
?>
<form method="POST" action="<?php echo admin_url('my-page'); ?>">
    <?php echo csrf_token_field(); ?>
    <input type="hidden" name="action" value="save">

    <div class="form-group">
        <label>Налаштування 1</label>
        <input type="text" name="setting1"
               value="<?php echo htmlspecialchars($settings['setting1'] ?? '', ENT_QUOTES, 'UTF-8'); ?>"
               class="form-control">
    </div>

    <button type="submit" class="btn btn-primary">Зберегти</button>
</form>
<?php
$content = ob_get_clean();

render_admin_layout([
    'title' => 'Моя сторінка',
    'content' => $content,
]);
```

### Таблиця

```php
$items = get_items();

ob_start();
?>
<div class="table-responsive">
    <table class="table">
        <thead>
            <tr>
                <th>ID</th>
                <th>Назва</th>
                <th>Дії</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($items as $item): ?>
                <tr>
                    <td><?php echo htmlspecialchars($item['id'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                    <td><?php echo htmlspecialchars($item['name'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                    <td>
                        <a href="<?php echo admin_url('my-page/edit?id=' . $item['id']); ?>"
                           class="btn btn-sm btn-primary">Редагувати</a>
                        <form method="POST" style="display: inline;">
                            <?php echo csrf_token_field(); ?>
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="id" value="<?php echo $item['id']; ?>">
                            <button type="submit" class="btn btn-sm btn-danger"
                                    onclick="return confirm('Ви впевнені?');">Видалити</button>
                        </form>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>
<?php
$content = ob_get_clean();

render_admin_layout([
    'title' => 'Моя сторінка',
    'content' => $content,
]);
```

### Модальне вікно

```php
ob_start();
?>
<!-- Кнопка для відкриття модального вікна -->
<button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#myModal">
    Відкрити модальне вікно
</button>

<!-- Модальне вікно -->
<div class="modal fade" id="myModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Підтвердження</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Ви впевнені, що хочете виконати цю дію?</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Скасувати</button>
                <button type="button" class="btn btn-primary">Підтвердити</button>
            </div>
        </div>
    </div>
</div>
<?php
$content = ob_get_clean();

render_admin_layout([
    'title' => 'Моя сторінка',
    'content' => $content,
]);
```

## Крок 6: Реєстрація маршруту

### Додавання маршруту

Додайте маршрут в `content/admin/includes/admin-routes.php`:

```php
$router->add(['GET', 'POST'], 'my-page', 'admin_my_page', [
    'middleware' => ['admin.auth']
]);
```

## Крок 7: Додавання в меню

### Реєстрація пункту меню

```php
// У плагіні або темі
add_filter('admin_menu', function (array $menu): array {
    $menu[] = [
        'text' => 'Мій розділ',
        'title' => 'Управління моїм розділом',
        'icon' => 'fas fa-cog',
        'href' => admin_url('my-page'),
        'page' => 'my-page',
        'permission' => 'admin.access',
        'order' => 50,
    ];
    return $menu;
});
```

## Повний приклад

```php
<?php

declare(strict_types=1);

function admin_my_page(): void
{
    // Обробка POST
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (isset($_POST['action'])) {
            // CSRF захист
            if (!verify_csrf_token($_POST['_token'] ?? '')) {
                set_flash_message('Помилка безпеки', 'error');
                admin_redirect('my-page');
                exit;
            }

            switch ($_POST['action']) {
                case 'save':
                    $setting1 = sanitize_input($_POST['setting1'] ?? '');
                    save_setting('setting1', $setting1);
                    set_flash_message('Налаштування збережено', 'success');
                    break;

                case 'delete':
                    $id = (int)($_POST['id'] ?? 0);
                    if ($id > 0) {
                        delete_item($id);
                        set_flash_message('Елемент видалено', 'success');
                    }
                    break;
            }

            admin_redirect('my-page');
            exit;
        }
    }

    // Отримання даних
    $settings = get_settings();
    $items = get_items();

    // Рендеринг
    ob_start();
    ?>
    <div class="card">
        <div class="card-header">
            <h3>Налаштування</h3>
        </div>
        <div class="card-body">
            <form method="POST" action="<?php echo admin_url('my-page'); ?>">
                <?php echo csrf_token_field(); ?>
                <input type="hidden" name="action" value="save">

                <div class="form-group">
                    <label>Налаштування 1</label>
                    <input type="text" name="setting1"
                           value="<?php echo htmlspecialchars($settings['setting1'] ?? '', ENT_QUOTES, 'UTF-8'); ?>"
                           class="form-control">
                </div>

                <button type="submit" class="btn btn-primary">Зберегти</button>
            </form>
        </div>
    </div>

    <div class="card mt-4">
        <div class="card-header">
            <h3>Елементи</h3>
        </div>
        <div class="card-body">
            <table class="table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Назва</th>
                        <th>Дії</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($items as $item): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($item['id'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($item['name'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td>
                                <form method="POST" style="display: inline;">
                                    <?php echo csrf_token_field(); ?>
                                    <input type="hidden" name="action" value="delete">
                                    <input type="hidden" name="id" value="<?php echo $item['id']; ?>">
                                    <button type="submit" class="btn btn-sm btn-danger"
                                            onclick="return confirm('Ви впевнені?');">Видалити</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    <?php
    $content = ob_get_clean();

    render_admin_layout([
        'title' => 'Моя сторінка',
        'pageBreadcrumbs' => [
            ['text' => 'Головна', 'href' => admin_url('dashboard')],
            ['text' => 'Моя сторінка'],
        ],
        'pageHeaderButtons' => [
            [
                'text' => 'Додати',
                'href' => admin_url('my-page/add'),
                'class' => 'btn-primary',
            ],
        ],
        'content' => $content,
    ]);
}
```

## Найкращі практики

### 1. POST-Redirect-GET

```php
// Завжди використовуйте редирект після POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    save_data();
    admin_redirect('my-page');
    exit;
}
```

### 2. CSRF захист

```php
// Завжди перевіряйте CSRF токени
if (!verify_csrf_token($_POST['_token'] ?? '')) {
    set_flash_message('Помилка безпеки', 'error');
    admin_redirect('my-page');
    exit;
}
```

### 3. Валідація даних

```php
// Завжди валідуйте вхідні дані
$setting1 = sanitize_input($_POST['setting1'] ?? '');
if (empty($setting1)) {
    set_flash_message('Поле обов\'язкове', 'error');
    admin_redirect('my-page');
    exit;
}
```

### 4. Екранування виводу

```php
// Завжди екрануйте вивід
echo htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
```

## Наступні кроки

- [Компоненти](Components) — використання компонентів
- [Кастомізація](Customization) — кастомізація адмін-панелі

---

**Сторінка адмінки створено! Тепер можна додавати функціональність!** 📄
