# Маршрутизація

Детальний опис системи маршрутизації в FLOWAXY-CMS.

## Огляд

Система маршрутизації обробляє HTTP запити та направляє їх до відповідних обробників (контроллерів, функцій).

## Основні поняття

### Маршрут (Route)

Маршрут — це зв'язок між URL та обробником запиту.

**Компоненти маршруту:**

-   **URL** — шлях запиту
-   **Метод** — HTTP метод (GET, POST, PUT, DELETE)
-   **Обробник** — функція або контроллер
-   **Middleware** — проміжні обробники
-   **Параметри** — динамічні частини URL

## Створення маршрутів

### Базовий маршрут

```php
use Flowaxy\Core\Routing\Router;

$router = new Router();

// Простий маршрут
$router->add('GET', '/', function() {
    return 'Головна сторінка';
});

// Маршрут з обробником
$router->add('GET', '/about', 'AboutController@index');
```

### Маршрути з параметрами

```php
// Параметр {id}
$router->add('GET', '/user/{id}', function($id) {
    return "Користувач #{$id}";
});

// Декілька параметрів
$router->add('GET', '/post/{slug}/{id}', function($slug, $id) {
    return "Пост: {$slug} (#{$id})";
});

// Опціональні параметри
$router->add('GET', '/page/{slug?}', function($slug = 'home') {
    return "Сторінка: {$slug}";
});
```

### HTTP методи

```php
// Один метод
$router->add('GET', '/users', 'UserController@index');
$router->add('POST', '/users', 'UserController@store');

// Декілька методів
$router->add(['GET', 'POST'], '/login', 'AuthController@login');

// Всі методи
$router->add('*', '/api', 'ApiController@handle');
```

## Обробники маршрутів

### Функції

```php
$router->add('GET', '/', function() {
    return 'Головна';
});

$router->add('GET', '/about', function() {
    return view('about');
});
```

### Контроллери

```php
// Метод контроллера
$router->add('GET', '/users', 'UserController@index');

// З параметрами
$router->add('GET', '/user/{id}', 'UserController@show');
```

### Класи

```php
// Клас з методом __invoke
$router->add('GET', '/api', ApiHandler::class);
```

## Middleware

Middleware — це проміжні обробники, які виконуються до або після обробника маршруту.

### Реєстрація middleware

```php
// Middleware для конкретного маршруту
$router->add('GET', '/admin', 'AdminController@index', [
    'middleware' => ['auth', 'admin'],
]);

// Глобальний middleware
$router->middleware('auth', function($request, $next) {
    if (!isLoggedIn()) {
        return redirect('/login');
    }
    return $next($request);
});
```

### Приклад middleware

```php
// Middleware для авторизації
$router->middleware('auth', function($request, $next) {
    if (!Auth::check()) {
        return redirect('/login');
    }
    return $next($request);
});

// Middleware для CSRF
$router->middleware('csrf', function($request, $next) {
    if ($request->isPost() && !verify_csrf_token($request->get('_token'))) {
        return response('CSRF token mismatch', 403);
    }
    return $next($request);
});
```

## Групування маршрутів

### Група з префіксом

```php
$router->group('/admin', function($router) {
    $router->add('GET', '/', 'AdminController@index');
    $router->add('GET', '/users', 'AdminController@users');
    $router->add('GET', '/settings', 'AdminController@settings');
});

// Результат:
// /admin → AdminController@index
// /admin/users → AdminController@users
// /admin/settings → AdminController@settings
```

### Група з middleware

```php
$router->group('/admin', function($router) {
    $router->add('GET', '/', 'AdminController@index');
}, ['middleware' => ['auth', 'admin']]);
```

### Вкладені групи

```php
$router->group('/api', function($router) {
    $router->group('/v1', function($router) {
        $router->add('GET', '/users', 'ApiController@users');
    });
});

// Результат: /api/v1/users
```

## Маршрути адмінки

Адмін-панель використовує окремий роутер з базовим шляхом `/admin`.

### Автоматична завантаження

Маршрути адмінки автоматично завантажуються з `content/admin/includes/admin-routes.php`:

```php
// content/admin/includes/admin-routes.php
$adminRouter = RouterManager::getAdminRouter();

$adminRouter->add('GET', '/', 'admin_dashboard_page');
$adminRouter->add('GET', '/plugins', 'admin_plugins_page');
$adminRouter->add('GET', '/themes', 'admin_themes_page');
```

### Функції сторінок

Сторінки адмінки — це функції без класів:

```php
// content/admin/Pages/dashboard.php
function admin_dashboard_page(): void
{
    render_admin_layout('dashboard', [
        'title' => 'Головна',
    ]);
}
```

## Маршрути плагінів

Плагіни можуть реєструвати власні маршрути через хуки:

```php
// content/plugins/social-networks/Plugin.php
public function boot(): void
{
    Action::add('register_routes', function($router) {
        $router->add('GET', '/social-networks', [$this, 'renderNetworks']);
    });
}
```

## Маршрути тем

Темі можуть реєструвати власні маршрути через `routes.php`:

```php
// content/themes/default/routes.php
return [
    ['GET', '/', 'home'],
    ['GET', '/about', 'about'],
    ['GET', '/contact', 'contact'],
];
```

## Обробка запитів

### Життєвий цикл запиту

```mermaid
graph TD
    A[HTTP Request] --> B[Router]
    B --> C{Route Found?}
    C -->|Yes| D[Middleware]
    C -->|No| E[404 Error]
    D --> F[Handler]
    F --> G[Response]
    G --> H[HTTP Response]
```

### Приклад обробки

```php
// 1. Запит приходить на /user/123
$request = new Request();

// 2. Роутер знаходить маршрут
$route = $router->match('GET', '/user/123');

// 3. Виконується middleware
foreach ($route->getMiddleware() as $middleware) {
    $request = $middleware($request, $next);
}

// 4. Викликається обробник
$response = $route->getHandler()($request, 123);

// 5. Повертається відповідь
return $response;
```

## Параметри маршрутів

### Отримання параметрів

```php
// Маршрут: /user/{id}/post/{postId}
$router->add('GET', '/user/{id}/post/{postId}', function($id, $postId) {
    return "Користувач #{$id}, Пост #{$postId}";
});
```

### Валідація параметрів

```php
// З регулярним виразом
$router->add('GET', '/user/{id}', function($id) {
    // $id буде тільки числом
}, ['where' => ['id' => '\d+']]);
```

## Найкращі практики

### 1. Використовуйте RESTful маршрути

```php
// Добре: RESTful
$router->add('GET', '/users', 'UserController@index');
$router->add('GET', '/users/{id}', 'UserController@show');
$router->add('POST', '/users', 'UserController@store');
$router->add('PUT', '/users/{id}', 'UserController@update');
$router->add('DELETE', '/users/{id}', 'UserController@destroy');

// Погано: не RESTful
$router->add('GET', '/get-users', 'UserController@getUsers');
$router->add('POST', '/create-user', 'UserController@createUser');
```

### 2. Групуйте пов'язані маршрути

```php
// Добре: групування
$router->group('/api/v1', function($router) {
    $router->add('GET', '/users', 'ApiController@users');
    $router->add('GET', '/posts', 'ApiController@posts');
});

// Погано: дублювання
$router->add('GET', '/api/v1/users', 'ApiController@users');
$router->add('GET', '/api/v1/posts', 'ApiController@posts');
```

### 3. Використовуйте middleware для загальної логіки

```php
// Добре: middleware
$router->group('/admin', function($router) {
    // Всі маршрути захищені
}, ['middleware' => ['auth', 'admin']]);

// Погано: перевірка в кожному обробнику
$router->add('GET', '/admin', function() {
    if (!isAdmin()) {
        return redirect('/login');
    }
    // ...
});
```

## Наступні кроки

-   [Система хуків](Hooks-System) — Actions та Filters
-   [Dependency Injection](Dependency-Injection) — DI контейнер
-   [Розробка плагінів](../Development/Plugins/Overview) — маршрути в плагінах

---

**Система маршрутизації забезпечує гнучку обробку запитів!** 🛣️
