# Система хуків

Детальний опис системи хуків (Actions та Filters) в FLOWAXY-CMS.

## Огляд

Система хуків дозволяє розширювати функціональність системи без зміни ядра. Вона натхненна системою хуків WordPress та підтримує два типи хуків:

- **Actions** — виконання дій (без повернення значення)
- **Filters** — фільтрація даних (з поверненням значення)

## Типи хуків

### Actions (Події)

Actions використовуються для виконання дій у певні моменти роботи системи.

**Характеристики:**
- Не повертають значення
- Виконуються послідовно
- Можуть мати пріоритет

**Приклад:**
```php
// Реєстрація action
Action::add('init', function() {
    echo 'Система ініціалізована';
});

// Виконання action
Action::do('init');
```

### Filters (Фільтри)

Filters використовуються для зміни даних перед їх використанням.

**Характеристики:**
- Повертають значення
- Можуть бути ланцюжковими (результат одного передається в наступний)
- Можуть мати пріоритет

**Приклад:**
```php
// Реєстрація filter
Filter::add('the_title', function($title) {
    return strtoupper($title);
});

// Застосування filter
$title = Filter::apply('the_title', 'Hello World');
// Результат: 'HELLO WORLD'
```

## Використання

### Реєстрація Actions

```php
use Flowaxy\Core\Hooks\Action;

// Простий action
Action::add('plugin_loaded', function($pluginName) {
    Log::info("Плагін {$pluginName} завантажено");
});

// Action з пріоритетом
Action::add('init', function() {
    // Виконається першим
}, 5); // Пріоритет 5 (вище за замовчуванням 10)

// Action з виконанням один раз
Action::add('setup', function() {
    // Виконається тільки один раз
}, 10, true); // $once = true
```

### Виконання Actions

```php
use Flowaxy\Core\Hooks\Action;

// Виконання action
Action::do('init');

// З параметрами
Action::do('plugin_loaded', ['social-networks']);

// Перевірка наявності
if (Action::has('init')) {
    Action::do('init');
}
```

### Реєстрація Filters

```php
use Flowaxy\Core\Hooks\Filter;

// Простий filter
Filter::add('the_title', function($title) {
    return strtoupper($title);
});

// Filter з пріоритетом
Filter::add('the_content', function($content) {
    return strip_tags($content);
}, 20); // Пріоритет 20 (нижче за замовчуванням 10)

// Ланцюжковий filter
Filter::add('the_content', function($content) {
    return nl2br($content);
});
```

### Застосування Filters

```php
use Flowaxy\Core\Hooks\Filter;

// Застосування filter
$title = Filter::apply('the_title', 'Hello World');

// З параметрами
$content = Filter::apply('the_content', $content, ['post_id' => 123]);

// Перевірка наявності
if (Filter::has('the_title')) {
    $title = Filter::apply('the_title', $title);
}
```

## Пріоритети

Пріоритет визначає порядок виконання хуків. Нижче пріоритет = вище пріоритет виконання.

**Стандартні пріоритети:**
- `1-5` — дуже високий пріоритет
- `6-9` — високий пріоритет
- `10` — стандартний пріоритет (за замовчуванням)
- `11-19` — низький пріоритет
- `20+` — дуже низький пріоритет

**Приклад:**
```php
// Виконається першим (пріоритет 5)
Action::add('init', function() {
    echo '1';
}, 5);

// Виконається другим (пріоритет 10)
Action::add('init', function() {
    echo '2';
}, 10);

// Виконається третім (пріоритет 20)
Action::add('init', function() {
    echo '3';
}, 20);

// Результат: '123'
Action::do('init');
```

## Використання в плагінах

### Реєстрація хуків з плагіна

```php
<?php
namespace Flowaxy\Plugins\SocialNetworks;

use Flowaxy\Support\Base\BasePlugin;
use Flowaxy\Core\Hooks\Action;
use Flowaxy\Core\Hooks\Filter;

final class Plugin extends BasePlugin
{
    public function boot(): void
    {
        // Реєстрація через метод плагіна
        $this->registerAction('init', [$this, 'onInit']);
        $this->registerFilter('theme_social_networks', [$this, 'getSocialNetworks']);

        // Або напряму
        Action::add('admin_menu', [$this, 'addAdminMenu']);
    }

    public function onInit(): void
    {
        // Логіка ініціалізації
    }

    public function getSocialNetworks(array $networks = []): array
    {
        // Додавання соціальних мереж
        $networks[] = [
            'name' => 'Facebook',
            'url' => 'https://facebook.com',
        ];

        return $networks;
    }
}
```

## Використання в темах

### Реєстрація хуків з теми

```php
<?php
namespace Flowaxy\Themes\Default;

use Flowaxy\Support\Base\BaseTheme;
use Flowaxy\Core\Hooks\Filter;

final class Theme extends BaseTheme
{
    public function boot(): void
    {
        // Фільтрація заголовка
        Filter::add('the_title', [$this, 'formatTitle']);
    }

    public function formatTitle(string $title): string
    {
        return '<h1>' . htmlspecialchars($title) . '</h1>';
    }
}
```

## Глобальні функції

Система надає глобальні функції для роботи з хуками:

```php
// Actions
add_action('hook_name', $callback, $priority = 10, $once = false);
do_action('hook_name', ...$args);
has_action('hook_name');

// Filters
add_filter('hook_name', $callback, $priority = 10);
apply_filters('hook_name', $value, ...$args);
has_filter('hook_name');
```

**Приклад:**
```php
// Реєстрація
add_action('init', function() {
    echo 'Init';
});

// Виконання
do_action('init');

// Filter
add_filter('the_title', function($title) {
    return strtoupper($title);
});

$title = apply_filters('the_title', 'Hello');
```

## Стандартні хуки системи

### Actions

- `init` — ініціалізація системи
- `plugins_loaded` — після завантаження плагінів
- `theme_loaded` — після завантаження теми
- `admin_init` — ініціалізація адмінки
- `admin_menu` — реєстрація меню адмінки
- `before_render` — перед рендерингом
- `after_render` — після рендерингу

### Filters

- `the_title` — фільтрація заголовка
- `the_content` — фільтрація контенту
- `theme_social_networks` — соціальні мережі
- `admin_menu_items` — пункти меню адмінки
- `theme_components` — компоненти теми

## Моніторинг продуктивності

Система хуків має вбудований моніторинг продуктивності:

```php
use Flowaxy\Core\Hooks\HookManager;

$hookManager = App::make(HookManagerInterface::class);
$stats = $hookManager->getCallStats();

// Статистика викликів хуків
foreach ($stats as $hookName => $count) {
    echo "{$hookName}: {$count} викликів\n";
}
```

## Найкращі практики

### 1. Використовуйте пріоритети правильно

```php
// Погано: всі з пріоритетом 10
Action::add('init', $callback1);
Action::add('init', $callback2);
Action::add('init', $callback3);

// Добре: з правильними пріоритетами
Action::add('init', $callback1, 5);   // Спочатку
Action::add('init', $callback2, 10);  // Потім
Action::add('init', $callback3, 20);  // В кінці
```

### 2. Перевіряйте наявність хуків

```php
// Перевірка перед виконанням
if (Action::has('custom_hook')) {
    Action::do('custom_hook');
}
```

### 3. Використовуйте Filters для зміни даних

```php
// Добре: використання filter
$title = Filter::apply('the_title', $title);

// Погано: пряма зміна
$title = strtoupper($title); // Не дозволяє іншим плагінам змінювати
```

### 4. Документуйте власні хуки

```php
/**
 * Action: plugin_loaded
 * Виконується після завантаження плагіна
 *
 * @param string $pluginName Назва плагіна
 */
Action::add('plugin_loaded', function($pluginName) {
    // ...
});
```

## Наступні кроки

- [Маршрутизація](Routing) — роутинг та маршрути
- [Dependency Injection](Dependency-Injection) — DI контейнер
- [Розробка плагінів](../Development/Plugins/Plugin-Hooks) — використання хуків у плагінах

---

**Система хуків забезпечує гнучкість та розширюваність!** 🎣
