# Dependency Injection

Детальний опис Dependency Injection контейнера в FLOWAXY-CMS.

## Огляд

DI контейнер управляє залежностями та життєвим циклом об'єктів. Він забезпечує автоматичну інжекцію залежностей та управління сервісами.

## Основні поняття

### Binding (Прив'язка)

Binding — це реєстрація залежності в контейнері.

**Типи binding:**
- **Singleton** — один екземпляр на весь життєвий цикл
- **Transient** — новий екземпляр при кожному запиті
- **Lazy** — відкладене створення

### Resolution (Резолвінг)

Resolution — це отримання об'єкта з контейнера з автоматичною інжекцією залежностей.

## Використання

### Базове використання

```php
use Flowaxy\Core\System\Container\Container;

$container = new Container();

// Реєстрація сервісу
$container->bind(LoggerInterface::class, Logger::class);

// Отримання сервісу
$logger = $container->make(LoggerInterface::class);
```

### Singleton

```php
// Реєстрація як singleton
$container->singleton(LoggerInterface::class, Logger::class);

// Перший виклик створює екземпляр
$logger1 = $container->make(LoggerInterface::class);

// Другий виклик повертає той самий екземпляр
$logger2 = $container->make(LoggerInterface::class);

// $logger1 === $logger2 (true)
```

### Transient

```php
// Реєстрація як transient
$container->bind(LoggerInterface::class, Logger::class);

// Кожен виклик створює новий екземпляр
$logger1 = $container->make(LoggerInterface::class);
$logger2 = $container->make(LoggerInterface::class);

// $logger1 !== $logger2 (true)
```

### Замикання (Closure)

```php
// Реєстрація через замикання
$container->bind(LoggerInterface::class, function($container) {
    return new Logger('/path/to/logs');
});
```

## Автоматична інжекція залежностей

### Через конструктор

```php
class UserService
{
    public function __construct(
        private LoggerInterface $logger,
        private DatabaseInterface $db
    ) {}
}

// Контейнер автоматично інжектує залежності
$userService = $container->make(UserService::class);
```

### Через метод

```php
class UserService
{
    public function setLogger(LoggerInterface $logger): void
    {
        $this->logger = $logger;
    }
}

// Виклик методу з інжекцією
$container->call([$userService, 'setLogger']);
```

## Service Providers

Service Providers реєструють сервіси в контейнері.

### Створення Service Provider

```php
use Flowaxy\Core\System\Container\ServiceProvider;

class AppServiceProvider extends ServiceProvider
{
    protected function registerBindings(): void
    {
        // Реєстрація сервісів
        $this->container->singleton(LoggerInterface::class, Logger::class);
        $this->container->bind(DatabaseInterface::class, Database::class);
    }

    protected function boot(): void
    {
        // Ініціалізація після реєстрації
    }
}
```

### Реєстрація Service Provider

```php
// Автоматично завантажується з flowaxy/Core/System/Container/Providers/
// Або вручну:
$container->registerServiceProvider(AppServiceProvider::class);
```

## Аліаси

Аліаси дозволяють використовувати короткі назви для сервісів.

```php
// Реєстрація з аліасом
$container->alias('logger', LoggerInterface::class);

// Використання аліасу
$logger = $container->make('logger');
```

## Теги сервісів

Теги дозволяють групувати сервіси для масової обробки.

```php
// Реєстрація з тегом
$container->tag([Logger::class, FileLogger::class], 'loggers');

// Отримання всіх сервісів з тегом
$loggers = $container->tagged('loggers');
```

## Lazy Loading

Lazy loading створює об'єкти тільки при першому зверненні.

```php
// Реєстрація як lazy
$container->bind(LoggerInterface::class, Logger::class, lazy: true);

// Об'єкт створюється тільки при першому виклику
$logger = $container->make(LoggerInterface::class);
```

## Перевірка наявності

```php
// Перевірка чи зареєстрований сервіс
if ($container->has(LoggerInterface::class)) {
    $logger = $container->make(LoggerInterface::class);
}
```

## Приклади використання

### Реєстрація сервісу

```php
use Flowaxy\Core\System\Container\Container;

$container = Container::getInstance();

// Singleton
$container->singleton(LoggerInterface::class, Logger::class);

// Transient
$container->bind(DatabaseInterface::class, Database::class);

// З замиканням
$container->bind(CacheInterface::class, function($container) {
    return new Cache($container->make(ConfigInterface::class));
});
```

### Отримання сервісу

```php
// Через make()
$logger = $container->make(LoggerInterface::class);

// Через resolve() (alias для make())
$logger = $container->resolve(LoggerInterface::class);

// Через фасад
use Flowaxy\Support\Facades\Log;
Log::info('Message');
```

### Виклик методів

```php
// Виклик функції з інжекцією
$result = $container->call(function(LoggerInterface $logger) {
    $logger->info('Called');
    return 'Done';
});

// Виклик методу з інжекцією
$result = $container->call([$service, 'method'], ['param' => 'value']);
```

## Найкращі практики

### 1. Використовуйте інтерфейси

```php
// Добре: через інтерфейс
$container->bind(LoggerInterface::class, Logger::class);

// Погано: конкретний клас
$container->bind(Logger::class, Logger::class);
```

### 2. Singleton для важких об'єктів

```php
// Добре: singleton для важких об'єктів
$container->singleton(DatabaseInterface::class, Database::class);

// Погано: transient для важких об'єктів
$container->bind(DatabaseInterface::class, Database::class);
```

### 3. Використовуйте Service Providers

```php
// Добре: через Service Provider
class AppServiceProvider extends ServiceProvider
{
    protected function registerBindings(): void
    {
        $this->container->singleton(LoggerInterface::class, Logger::class);
    }
}

// Погано: реєстрація вручну скрізь
$container->singleton(LoggerInterface::class, Logger::class);
```

## Наступні кроки

- [Структура ядра](Core-Structure) — детальний опис `flowaxy/`
- [Система хуків](Hooks-System) — Actions та Filters
- [Маршрутизація](Routing) — роутинг та маршрути

---

**DI контейнер забезпечує управління залежностями!** 💉
