# Структура контенту (content/)

Детальний опис структури ізольованого контенту FLOWAXY-CMS.

## Загальний огляд

Папка `content/` містить **ізольований контент** системи. Вона повністю незалежна від ядра і використовує тільки функції та масиви (БЕЗ класів та namespace).

## Принцип ізоляції

### Правила content/

-   ✅ **Тільки функції** — без класів
-   ✅ **Тільки масиви** — конфігурації через масиви
-   ✅ **Без namespace** — глобальний простір імен
-   ✅ **Взаємодія через хуки** — Actions та Filters
-   ✅ **Хелпери** — використання функцій-хелперів з ядра

### Заборонено в content/

-   ❌ Класи з namespace
-   ❌ Пряме використання класів ядра
-   ❌ Залежності від структури ядра

## Структура content/

```
content/
├── admin/            # Адмін-панель
│   ├── Components/   # Компоненти конструктора (масиви)
│   ├── Pages/        # Сторінки адмінки (функції)
│   ├── Templates/    # Шаблони (layout.php, login.php)
│   └── includes/     # Допоміжні файли
│
├── plugins/          # Плагіни
│   └── [plugin-name]/
│       ├── Plugin.php        # Клас плагіна (успадковує BasePlugin)
│       ├── admin/            # Адмін-інтерфейс
│       └── config/           # Конфігурація плагіна
│
├── themes/           # Темі
│   └── [theme-name]/
│       ├── Theme.php         # Клас теми (успадковує BaseTheme)
│       ├── components/        # Компоненти теми
│       ├── widgets/           # Віджети теми
│       ├── layouts/           # Макети теми
│       └── templates/         # Шаблони теми
│
└── support/          # Допоміжні скрипти
```

## admin/ — Адмін-панель

**Призначення:** Адміністративна панель системи.

### Components/ — Компоненти конструктора

Компоненти визначаються через масиви конфігурації.

**Структура:**

```
admin/Components/
├── Button/          # Кнопки (масиви конфігурації)
├── Form/            # Форми (масиви конфігурації)
├── Modal/           # Модальні вікна (масиви конфігурації)
├── Notification/     # Сповіщення (масиви конфігурації)
├── Menu/             # Меню (масиви конфігурації)
├── Header/           # Шапка (масиви конфігурації)
├── Footer/           # Футер (масиви конфігурації)
├── Layout/           # Layout компоненти (масиви конфігурації)
└── Sidebar/          # Бічна панель (масиви конфігурації)
```

**Приклад кнопки:**

```php
// admin/Components/Button/button.php
return [
    'text' => 'Зберегти',
    'type' => 'submit',
    'class' => 'btn-primary',
    'onclick' => 'saveForm()',
];
```

### Pages/ — Сторінки адмінки

Сторінки адмінки — це функції без класів.

**Структура:**

```
admin/Pages/
├── dashboard.php        # Головна сторінка
├── login.php            # Вхід
├── logout.php           # Вихід
├── settings.php         # Налаштування
├── plugins.php          # Плагіни
├── themes.php           # Темі
└── ...
```

**Приклад сторінки:**

```php
// admin/Pages/dashboard.php
function admin_dashboard_page(): void
{
    // Використання хелперів
    $data = db_query('SELECT * FROM users');

    // Рендеринг через компоненти
    render_admin_layout('dashboard', [
        'users' => $data,
    ]);
}
```

### Templates/ — Шаблони

Шаблони для адмін-панелі.

**Файли:**

-   `layout.php` — основний макет
-   `login.php` — сторінка входу

### includes/ — Допоміжні файли

**Файли:**

-   `admin-routes.php` — реєстрація маршрутів адмінки
-   `functions.php` — допоміжні функції
-   `components.php` — завантаження компонентів

## plugins/ — Плагіни

**Призначення:** Модульні розширення функціональності.

### Структура плагіна

```
content/plugins/[plugin-name]/
├── Plugin.php           # Клас плагіна (успадковує BasePlugin)
├── admin/               # Адмін-інтерфейс
│   └── pages/           # Сторінки адмінки плагіна
├── config/              # Конфігурація
│   ├── plugin.ini       # Метадані плагіна (опціонально)
│   └── [config].ini     # Конфігураційні файли
└── README.md            # Документація
```

### Plugin.php

Клас плагіна успадковує `BasePlugin`:

```php
<?php
namespace Flowaxy\Plugins\SocialNetworks;

use Flowaxy\Support\Base\BasePlugin;

final class Plugin extends BasePlugin
{
    public function boot(): void
    {
        // Реєстрація хуків
        $this->registerFilter('theme_social_networks', [$this, 'getSocialNetworks']);
    }

    public function getSocialNetworks(array $networks = []): array
    {
        // Логіка плагіна
        return $networks;
    }
}
```

### Життєвий цикл плагіна

1. **Install** — встановлення
2. **Activate** — активація
3. **Deactivate** — деактивація
4. **Uninstall** — видалення

**Детальніше:** [Розробка плагінів](../Development/Plugins/Overview)

## themes/ — Темі

**Призначення:** Інтерфейс користувача.

### Структура теми

```
content/themes/[theme-name]/
├── Theme.php            # Клас теми (успадковує BaseTheme)
├── components/          # Компоненти теми
│   └── [component]/
│       ├── component.php
│       ├── component.css
│       └── component.js
├── widgets/            # Віджети теми
│   └── [widget]/
│       ├── widget.php
│       ├── widget.css
│       └── responsive.css
├── layouts/             # Макети теми
│   └── main.php
├── templates/           # Шаблони теми
│   └── default.php
├── resources/           # Ресурси теми
│   ├── styles/          # CSS файли
│   ├── scripts/         # JS файли
│   ├── fonts/           # Шрифти
│   └── library/         # Бібліотеки
└── README.md            # Документація
```

### Theme.php

Клас теми успадковує `BaseTheme`:

```php
<?php
namespace Flowaxy\Themes\Default;

use Flowaxy\Support\Base\BaseTheme;

final class Theme extends BaseTheme
{
    public function boot(): void
    {
        // Реєстрація компонентів
        $this->registerComponent('slider');
        $this->registerComponent('auth');
    }

    public function renderTheme(): void
    {
        $this->render('default');
    }
}
```

### Компоненти теми

Компоненти — це переіспользувані блоки з власними стилями та скриптами.

**Структура:**

```
components/[component-name]/
├── component.php    # HTML та PHP логіка
├── component.css    # Стилі компонента
└── component.js     # JavaScript компонента
```

**Використання:**

```php
// У шаблоні теми
<?php theme_component('slider'); ?>
```

### Віджети теми

Віджети — це блоки контенту з адаптивними стилями.

**Структура:**

```
widgets/[widget-name]/
├── widget.php        # HTML та PHP логіка
├── widget.css        # Стилі віджета
└── responsive.css    # Адаптивні стилі
```

**Використання:**

```php
// У шаблоні теми
<?php theme_widget('marketplace', $data); ?>
```

### Макети та шаблони

**Layouts** — структура сторінок (header, footer, sidebars).

**Templates** — конкретні сторінки (default.php, single.php).

**Приклад:**

```php
// layouts/main.php
<!DOCTYPE html>
<html>
<head>
    <title><?php echo $pageTitle; ?></title>
</head>
<body>
    <?php echo Theme::yield('content'); ?>
</body>
</html>

// templates/default.php
<?php Theme::extend('main', ['content' => function() { ?>
    <h1>Головна сторінка</h1>
<?php }]); ?>
```

**Детальніше:** [Розробка тем](../Development/Themes/Overview)

## Взаємодія з ядром

### Хелпери

Ядро надає функції-хелпери для роботи з контентом:

```php
// Робота з БД
db_query('SELECT * FROM users');
db_fetch_all($result);
db_fetch_one($result);

// Безпека
csrf_token();
sanitize_input($data);
verify_csrf_token($token);

// Темі
theme_component('slider');
theme_asset('css/style.css');

// Адмінка
render_admin_layout('page', $data);
admin_url('/plugins');
```

### Хуки

Контент взаємодіє з ядром через хуки:

```php
// Actions
Action::add('init', function() {
    // Ініціалізація
});

// Filters
Filter::add('the_title', function($title) {
    return strtoupper($title);
});
```

## Автоматична обробка

### Плагіни

Ядро автоматично:

1. Сканує `content/plugins/`
2. Завантажує `Plugin.php`
3. Викликає `boot()` метод
4. Реєструє хуки

### Темі

Ядро автоматично:

1. Сканує `content/themes/`
2. Завантажує `Theme.php`
3. Викликає `boot()` метод
4. Реєструє компоненти та віджети

### Адмінка

Ядро автоматично:

1. Завантажує `admin/includes/admin-routes.php`
2. Реєструє маршрути
3. Завантажує сторінки з `admin/Pages/`
4. Рендерить через компоненти

## Наступні кроки

-   [Структура ядра](Core-Structure) — детальний опис `flowaxy/`
-   [Система хуків](Hooks-System) — Actions та Filters
-   [Розробка плагінів](../Development/Plugins/Overview)
-   [Розробка тем](../Development/Themes/Overview)

---

**Контент ізольований від ядра та легко розширюється!** 📦
