# Security API

Детальний опис API безпеки FLOWAXY-CMS.

## Огляд

Security API надає методи для забезпечення безпеки: CSRF захист, шифрування, хешування, санитизація, валідація.

## CSRF захист

### SecurityHelper

```php
use Flowaxy\Support\Helpers\SecurityHelper;

// Генерація CSRF токену
$token = SecurityHelper::csrfToken();

// Перевірка CSRF токену
if (SecurityHelper::verifyCsrfToken($token)) {
    // Безпечно
}
```

### Глобальні функції

```php
// Генерація токену
$token = csrf_token();

// Перевірка токену
if (verify_csrf_token($token)) {
    // Безпечно
}

// Поле для форми
echo csrf_token_field();
```

### Використання в формах

```php
<form method="POST">
    <?php echo csrf_token_field(); ?>
    <!-- Поля форми -->
</form>
```

## Шифрування

### Encryption Facade

```php
use Flowaxy\Support\Facades\Encryption;

// Шифрування
$encrypted = Encryption::encrypt('sensitive data');

// Розшифрування
$decrypted = Encryption::decrypt($encrypted);
```

## Хешування

### Hash Facade

```php
use Flowaxy\Support\Facades\Hash;

// Хешування пароля
$hash = Hash::make('password');

// Перевірка пароля
if (Hash::check('password', $hash)) {
    // Пароль правильний
}

// Перевірка чи потрібно перехешувати
if (Hash::needsRehash($hash)) {
    $hash = Hash::make('password');
}
```

## Санитизація

### SecurityHelper

```php
use Flowaxy\Support\Helpers\SecurityHelper;

// Санитизація вводу
$clean = SecurityHelper::sanitize($input);
```

### SanitizationHelper

```php
use Flowaxy\Support\Helpers\SanitizationHelper;

// Санитизація рядка
$clean = SanitizationHelper::sanitizeString($input);

// Санитизація HTML
$clean = SanitizationHelper::sanitizeHtml($html);

// Санитизація URL
$clean = SanitizationHelper::sanitizeUrl($url);

// Санитизація email
$clean = SanitizationHelper::sanitizeEmail($email);
```

### Глобальна функція

```php
// Санитизація вводу
$clean = sanitize_input($input);
```

## Валідація

### Validator Facade

```php
use Flowaxy\Support\Facades\Validator;

// Валідація даних
$validator = Validator::make($data, [
    'email' => 'required|email',
    'password' => 'required|min:8',
]);

if ($validator->fails()) {
    $errors = $validator->errors();
}
```

### ValidationHelper

```php
use Flowaxy\Support\Helpers\ValidationHelper;

// Валідація email
if (ValidationHelper::email($email)) {
    // Валідний email
}

// Валідація URL
if (ValidationHelper::url($url)) {
    // Валідний URL
}

// Валідація числа
if (ValidationHelper::numeric($value)) {
    // Валідне число
}
```

### SecurityHelper

```php
use Flowaxy\Support\Helpers\SecurityHelper;

// Валідація даних
if (SecurityHelper::validate($data, $rules)) {
    // Валідно
}
```

## Rate Limiting

### RateLimiter Facade

```php
use Flowaxy\Support\Facades\RateLimiter;

// Перевірка ліміту
if (RateLimiter::tooManyAttempts('key', 5)) {
    // Забагато спроб
}

// Збільшення лічильника
RateLimiter::hit('key', 60); // 60 секунд

// Отримання кількості спроб
$attempts = RateLimiter::attempts('key');

// Очищення
RateLimiter::clear('key');
```

## Content Security Policy

### CspHeaders

```php
use Flowaxy\Infrastructure\Security\CspHeaders;

// Створення CSP
$csp = CspHeaders::createDefault();

// Додавання джерел
$csp->addSources('script-src', ['https://cdn.example.com']);

// Генерація заголовка
$header = $csp->generate();
```

## Приклади використання

### CSRF захист в формах

```php
// У формі
<form method="POST" action="<?php echo admin_url('settings'); ?>">
    <?php echo csrf_token_field(); ?>
    <input type="text" name="setting1">
    <button type="submit">Зберегти</button>
</form>

// У обробнику
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verify_csrf_token($_POST['_token'] ?? '')) {
        set_flash_message('Помилка безпеки', 'error');
        admin_redirect('settings');
        exit;
    }
    // Обробка форми
}
```

### Хешування паролів

```php
use Flowaxy\Support\Facades\Hash;

// При реєстрації
$hashedPassword = Hash::make($password);

// При авторизації
if (Hash::check($password, $hashedPassword)) {
    // Пароль правильний
}
```

### Шифрування даних

```php
use Flowaxy\Support\Facades\Encryption;

// Шифрування
$encrypted = Encryption::encrypt('sensitive data');

// Розшифрування
$decrypted = Encryption::decrypt($encrypted);
```

### Санитизація вводу

```php
// У обробнику форм
$cleanInput = sanitize_input($_POST['input'] ?? '');

// Або через хелпер
use Flowaxy\Support\Helpers\SanitizationHelper;
$cleanInput = SanitizationHelper::sanitizeString($_POST['input'] ?? '');
```

### Валідація даних

```php
use Flowaxy\Support\Facades\Validator;

$validator = Validator::make($_POST, [
    'email' => 'required|email',
    'password' => 'required|min:8',
]);

if ($validator->fails()) {
    $errors = $validator->errors();
    // Обробка помилок
}
```

### Rate Limiting

```php
use Flowaxy\Support\Facades\RateLimiter;

// Перевірка перед дією
if (RateLimiter::tooManyAttempts('login_' . $ip, 5)) {
    set_flash_message('Забагато спроб входу', 'error');
    return;
}

// Збільшення лічильника
RateLimiter::hit('login_' . $ip, 300); // 5 хвилин
```

## Найкращі практики

### 1. Завжди використовуйте CSRF захист

```php
// Добре: CSRF захист
<form method="POST">
    <?php echo csrf_token_field(); ?>
    <!-- Поля -->
</form>

// Погано: Без CSRF захисту
<form method="POST">
    <!-- Поля -->
</form>
```

### 2. Санитуйте всі вхідні дані

```php
// Добре: Санитизація
$clean = sanitize_input($_POST['input'] ?? '');

// Погано: Без санитизації
$input = $_POST['input'];
```

### 3. Валідуйте дані перед збереженням

```php
// Добре: Валідація
$validator = Validator::make($data, $rules);
if ($validator->fails()) {
    // Обробка помилок
}

// Погано: Без валідації
save_data($data);
```

### 4. Хешуйте паролі

```php
// Добре: Хешування
$hash = Hash::make($password);

// Погано: Без хешування
$password = $password; // НЕ РОБІТЬ ТАК!
```

## Наступні кроки

- [Фасади](Facades) — всі фасади системи
- [Хелпери](Helpers) — всі хелпери системи

---

**Security API забезпечує безпеку системи!** 🔒
