# Plugin API

Детальний опис API для роботи з плагінами.

## Огляд

Plugin API надає методи для роботи з плагінами: життєвий цикл, налаштування, хуки, маршрути.

## BasePlugin

Базовий клас для всіх плагінів.

```php
use Flowaxy\Support\Base\BasePlugin;

class Plugin extends BasePlugin
{
    // Реєстрація хуків
    public function registerHooks(): void
    {
        $this->registerAction('init', [$this, 'onInit']);
        $this->registerFilter('the_title', [$this, 'formatTitle']);
    }

    // Реєстрація маршрутів
    public function registerRoutes(): void
    {
        $this->registerHook('admin_routes', function ($router) {
            $router->add(['GET', 'POST'], 'my-plugin', 'admin_my_plugin_page');
        });
    }

    // Життєвий цикл
    public function install(): void { }
    public function activate(): void { }
    public function deactivate(): void { }
    public function uninstall(): void { }
}
```

## Методи BasePlugin

### registerHooks()

Реєстрація хуків плагіна.

```php
public function registerHooks(): void
{
    // Реєстрація Action
    $this->registerAction('hook_name', $callback, $priority = 10);

    // Реєстрація Filter
    $this->registerFilter('hook_name', $callback, $priority = 10);
}
```

### registerRoutes()

Реєстрація маршрутів плагіна.

```php
public function registerRoutes(): void
{
    $this->registerHook('admin_routes', function ($router) {
        $router->add(['GET', 'POST'], 'my-plugin', 'admin_my_plugin_page');
    });
}
```

### getSettings()

Отримання налаштувань плагіна.

```php
// Отримання всіх налаштувань
$settings = $this->getSettings();

// Отримання конкретного налаштування
$value = $settings['key'] ?? 'default';
```

### setSetting()

Збереження налаштування плагіна.

```php
// Збереження одного налаштування
$this->setSetting('key', 'value');

// Збереження кількох налаштувань
foreach ($data as $key => $value) {
    $this->setSetting($key, $value);
}
```

### getSlug()

Отримання slug плагіна.

```php
$slug = $this->getSlug(); // 'my-plugin'
```

### getPluginDir()

Отримання директорії плагіна.

```php
$dir = $this->getPluginDir(); // '/path/to/content/plugins/my-plugin/'
```

## Фасад Plugin

```php
use Flowaxy\Support\Facades\Plugin;

// Отримання плагіна
$plugin = Plugin::get('plugin-slug');

// Перевірка активності
if (Plugin::isActive('plugin-slug')) {
    // Плагін активний
}

// Отримання всіх плагінів
$plugins = Plugin::all();
```

## Життєвий цикл

### install()

Викликається при встановленні плагіна.

```php
public function install(): void
{
    parent::install();

    // Створення таблиць БД
    // Ініціалізація конфігурацій
}
```

### activate()

Викликається при активації плагіна.

```php
public function activate(): void
{
    parent::activate();

    // Виконання міграцій
    // Ініціалізація даних
}
```

### deactivate()

Викликається при деактивації плагіна.

```php
public function deactivate(): void
{
    parent::deactivate();

    // Очищення тимчасових даних
}
```

### uninstall()

Викликається при видаленні плагіна.

```php
public function uninstall(): void
{
    parent::uninstall();

    // Видалення таблиць БД
    // Видалення конфігурацій
}
```

## Реєстрація хуків

### Через методи плагіна

```php
public function registerHooks(): void
{
    // Action
    $this->registerAction('init', [$this, 'onInit']);

    // Filter
    $this->registerFilter('the_title', [$this, 'formatTitle']);
}
```

### Через замикання

```php
public function registerHooks(): void
{
    $this->registerAction('init', function() {
        // Логіка
    });

    $this->registerFilter('admin_menu', function(array $menu): array {
        $menu[] = ['text' => 'Мій плагін', 'href' => admin_url('my-plugin')];
        return $menu;
    });
}
```

## Реєстрація маршрутів

```php
public function registerRoutes(): void
{
    $this->registerHook('admin_routes', function ($router) {
        $pageFile = __DIR__ . DS . 'admin' . DS . 'pages' . DS . 'my-plugin.php';
        if (file_exists($pageFile)) {
            require_once $pageFile;
        }

        if ($router && method_exists($router, 'add')) {
            $router->add(['GET', 'POST'], 'my-plugin', 'admin_my_plugin_page', [
                'middleware' => ['admin.auth']
            ]);
        }
    });
}
```

## Налаштування плагіна

### Отримання налаштувань

```php
// Всі налаштування
$settings = $this->getSettings();

// Конкретне налаштування
$value = $settings['key'] ?? 'default';
```

### Збереження налаштувань

```php
// Одне налаштування
$this->setSetting('key', 'value');

// Кілька налаштувань
$this->setSetting('key1', 'value1');
$this->setSetting('key2', 'value2');
```

Налаштування зберігаються в `storage/config/plugins/{plugin-slug}.ini`.

## Наступні кроки

- [Фасади](Facades) — всі фасади системи
- [Hooks API](Hooks-API) — API для роботи з хуками
- [Theme API](Theme-API) — API для роботи з темами

---

**Plugin API забезпечує повний контроль над плагінами!** 🔌
