# Hooks API

Детальний опис API для роботи з хуками (Actions та Filters).

## Огляд

Система хуків дозволяє розширювати функціональність без зміни ядра. Підтримуються два типи хуків: Actions та Filters.

## Класи для роботи з хуками

### Action

Клас для роботи з Actions.

```php
use Flowaxy\Core\Hooks\Action;

// Реєстрація action
Action::add('hook_name', $callback, $priority = 10, $once = false);

// Виконання action
Action::do('hook_name', ...$args);

// Перевірка наявності
if (Action::has('hook_name')) {
    Action::do('hook_name');
}
```

### Filter

Клас для роботи з Filters.

```php
use Flowaxy\Core\Hooks\Filter;

// Реєстрація filter
Filter::add('hook_name', $callback, $priority = 10);

// Застосування filter
$result = Filter::apply('hook_name', $value, ...$args);

// Перевірка наявності
if (Filter::has('hook_name')) {
    $result = Filter::apply('hook_name', $value);
}
```

## Фасад Hooks

```php
use Flowaxy\Support\Facades\Hooks;

// Реєстрація Action
Hooks::on('hook_name', $callback, $priority = 10);

// Виконання Action
Hooks::dispatch('hook_name', ...$args);

// Реєстрація Filter
Hooks::filter('hook_name', $callback, $priority = 10);

// Застосування Filter
$result = Hooks::apply('hook_name', $value, ...$args);
```

## Глобальні функції

### Actions

```php
// Реєстрація
add_action('hook_name', $callback, $priority = 10, $once = false);

// Виконання
do_action('hook_name', ...$args);

// Перевірка наявності
has_action('hook_name');
```

### Filters

```php
// Реєстрація
add_filter('hook_name', $callback, $priority = 10);

// Застосування
apply_filters('hook_name', $value, ...$args);

// Перевірка наявності
has_filter('hook_name');
```

## Стандартні хуки системи

### Actions

- `init` — ініціалізація системи
- `plugins_loaded` — після завантаження плагінів
- `theme_loaded` — після завантаження теми
- `admin_init` — ініціалізація адмінки
- `admin_menu` — реєстрація меню адмінки
- `before_render` — перед рендерингом
- `after_render` — після рендерингу

### Filters

- `admin_menu` — пункти меню адмінки
- `the_title` — заголовок
- `the_content` — контент
- `theme_social_networks` — соціальні мережі
- `theme_components` — компоненти теми

## Приклади використання

### Реєстрація Action

```php
// Через клас
Action::add('init', function() {
    echo 'System initialized';
});

// Через фасад
Hooks::on('init', function() {
    echo 'System initialized';
});

// Через глобальну функцію
add_action('init', function() {
    echo 'System initialized';
});
```

### Виконання Action

```php
// Через клас
Action::do('init');

// Через фасад
Hooks::dispatch('init');

// Через глобальну функцію
do_action('init');
```

### Реєстрація Filter

```php
// Через клас
Filter::add('the_title', function($title) {
    return strtoupper($title);
});

// Через фасад
Hooks::filter('the_title', function($title) {
    return strtoupper($title);
});

// Через глобальну функцію
add_filter('the_title', function($title) {
    return strtoupper($title);
});
```

### Застосування Filter

```php
// Через клас
$title = Filter::apply('the_title', 'Hello');

// Через фасад
$title = Hooks::apply('the_title', 'Hello');

// Через глобальну функцію
$title = apply_filters('the_title', 'Hello');
```

## Пріоритети

Пріоритет визначає порядок виконання. Нижче пріоритет = вище пріоритет виконання.

```php
// Високий пріоритет (виконується першим)
Action::add('init', $callback1, 5);

// Стандартний пріоритет
Action::add('init', $callback2, 10);

// Низький пріоритет (виконується останнім)
Action::add('init', $callback3, 20);
```

## Наступні кроки

- [Фасади](Facades) — всі фасади системи
- [Theme API](Theme-API) — API для роботи з темами
- [Plugin API](Plugin-API) — API для роботи з плагінами

---

**Hooks API забезпечує гнучкість та розширюваність!** 🎣
