# Хелпери

Повний довідник всіх хелперів FLOWAXY-CMS (44+ хелперів).

## Огляд

Хелпери — це допоміжні класи та функції для спрощення роботи з системою. Всі хелпери знаходяться в `flowaxy/Support/Helpers/`.

## Категорії хелперів

### Робота з БД

#### DatabaseHelper

Робота з базою даних.

```php
use Flowaxy\Support\Helpers\DatabaseHelper;

// Отримання підключення
$pdo = DatabaseHelper::getConnection();

// Виконання запиту
$result = DatabaseHelper::query('SELECT * FROM users');

// Отримання одного рядка
$user = DatabaseHelper::fetchOne('SELECT * FROM users WHERE id = ?', [1]);

// Отримання всіх рядків
$users = DatabaseHelper::fetchAll('SELECT * FROM users');
```

#### QueryHelper

Побудова запитів.

```php
use Flowaxy\Support\Helpers\QueryHelper;

// Побудова SELECT запиту
$query = QueryHelper::select('users', ['name', 'email'])
    ->where('status', 'active')
    ->orderBy('name')
    ->build();
```

### Робота з кешем

#### CacheHelper

Робота з кешем.

```php
use Flowaxy\Support\Helpers\CacheHelper;

// Отримання з кешу
$value = CacheHelper::get('key', 'default');

// Збереження в кеш
CacheHelper::put('key', 'value', 3600);

// Збереження з callback
$value = CacheHelper::remember('key', function() {
    return expensive_operation();
}, 3600);

// Видалення з кешу
CacheHelper::forget('key');
```

### Безпека

#### SecurityHelper

Безпека.

```php
use Flowaxy\Support\Helpers\SecurityHelper;

// CSRF токен
$token = SecurityHelper::csrfToken();

// Перевірка CSRF
if (SecurityHelper::verifyCsrfToken($token)) {
    // Безпечно
}

// Санитизація
$clean = SecurityHelper::sanitize($input);

// Валідація
if (SecurityHelper::validate($data, $rules)) {
    // Валідно
}
```

#### SanitizationHelper

Санитизація даних.

```php
use Flowaxy\Support\Helpers\SanitizationHelper;

// Санитизація рядка
$clean = SanitizationHelper::sanitizeString($input);

// Санитизація HTML
$clean = SanitizationHelper::sanitizeHtml($html);

// Санитизація URL
$clean = SanitizationHelper::sanitizeUrl($url);
```

#### ValidationHelper

Валідація даних.

```php
use Flowaxy\Support\Helpers\ValidationHelper;

// Валідація email
if (ValidationHelper::email($email)) {
    // Валідний email
}

// Валідація URL
if (ValidationHelper::url($url)) {
    // Валідний URL
}
```

### Робота з файлами

#### FileHelper

Робота з файлами.

```php
use Flowaxy\Support\Helpers\FileHelper;

// Читання файлу
$content = FileHelper::read('path/to/file.txt');

// Запис файлу
FileHelper::write('path/to/file.txt', 'content');

// Перевірка існування
if (FileHelper::exists('path/to/file.txt')) {
    // Файл існує
}

// Видалення
FileHelper::delete('path/to/file.txt');

// Розширення файлу
$ext = FileHelper::extension('file.txt'); // 'txt'
```

#### PathHelper

Робота з шляхами.

```php
use Flowaxy\Support\Helpers\PathHelper;

// Об'єднання шляхів
$path = PathHelper::join('dir1', 'dir2', 'file.txt');

// Нормалізація шляху
$path = PathHelper::normalize('dir1/../dir2/file.txt');

// Відносний шлях
$relative = PathHelper::relative('/base/path', '/base/path/to/file.txt');
```

### Робота з конфігурацією

#### ConfigHelper

Конфігурація.

```php
use Flowaxy\Support\Helpers\ConfigHelper;

// Отримання значення
$value = ConfigHelper::get('key', 'default');

// Встановлення значення
ConfigHelper::set('key', 'value');

// Завантаження з файлу
$config = ConfigHelper::load('path/to/config.php');
```

#### IniHelper

Робота з INI файлами.

```php
use Flowaxy\Support\Helpers\IniHelper;

// Читання INI
$data = IniHelper::readFile('path/to/file.ini', true); // true для секцій

// Запис INI
IniHelper::writeFile('path/to/file.ini', $data, true);
```

#### JsonHelper

Робота з JSON.

```php
use Flowaxy\Support\Helpers\JsonHelper;

// Читання JSON
$data = JsonHelper::readFile('path/to/file.json');

// Запис JSON
JsonHelper::writeFile('path/to/file.json', $data);

// Декодування
$data = JsonHelper::decode($jsonString);

// Кодування
$json = JsonHelper::encode($data);
```

### Робота з темами

#### ThemeHelper

Робота з темами.

```php
// Глобальні функції
theme_component('slider');
theme_asset('css/style.css');
```

#### ThemeMetadataHelper

Метадані тем.

```php
use Flowaxy\Support\Helpers\ThemeMetadataHelper;

// Читання метаданих
$metadata = ThemeMetadataHelper::readMetadata('theme-slug');
```

### Робота з плагінами

#### PluginMetadataHelper

Метадані плагінів.

```php
use Flowaxy\Support\Helpers\PluginMetadataHelper;

// Читання метаданих
$metadata = PluginMetadataHelper::readMetadata('plugin-slug');
```

### Адмін-панель

#### AdminHelper

Адмін-панель.

```php
// Глобальні функції
admin_url('page');
admin_redirect('page');
render_admin_layout($config);
```

#### ButtonHelper

Кнопки адмінки.

```php
use Flowaxy\Support\Helpers\ButtonHelper;

// Рендеринг кнопки
echo ButtonHelper::render([
    'text' => 'Зберегти',
    'type' => 'submit',
    'class' => 'btn-primary',
]);
```

#### FormHelper

Форми адмінки.

```php
use Flowaxy\Support\Helpers\FormHelper;

// Рендеринг форми
echo FormHelper::render([
    'action' => admin_url('settings'),
    'method' => 'POST',
    'fields' => [
        ['type' => 'text', 'name' => 'setting1', 'label' => 'Налаштування 1'],
    ],
]);
```

#### ModalHelper

Модальні вікна адмінки.

```php
use Flowaxy\Support\Helpers\ModalHelper;

// Рендеринг модального вікна
echo ModalHelper::render([
    'id' => 'my-modal',
    'title' => 'Підтвердження',
    'content' => 'Ви впевнені?',
]);
```

#### NotificationHelper

Сповіщення адмінки.

```php
use Flowaxy\Support\Helpers\NotificationHelper;

// Рендеринг сповіщення
echo NotificationHelper::render([
    'type' => 'success',
    'message' => 'Дані збережено',
]);
```

### Робота з рядками

#### StringHelper

Робота з рядками.

```php
use Flowaxy\Support\Helpers\StringHelper;

// Обрізання рядка
$short = StringHelper::truncate('Long string', 10);

// Генерація slug
$slug = StringHelper::slug('Hello World'); // 'hello-world'

// Камел кейс
$camel = StringHelper::camel('hello_world'); // 'helloWorld'
```

### Робота з числами

#### NumberHelper

Робота з числами.

```php
use Flowaxy\Support\Helpers\NumberHelper;

// Форматування числа
$formatted = NumberHelper::format(1234.56, 2); // '1,234.56'

// Округлення
$rounded = NumberHelper::round(123.456, 2); // 123.46
```

### Робота з датами

#### DateHelper

Робота з датами.

```php
use Flowaxy\Support\Helpers\DateHelper;

// Форматування дати
$formatted = DateHelper::format($date, 'Y-m-d H:i:s');

// Відносна дата
$relative = DateHelper::ago($date); // '2 години тому'
```

#### TimezoneHelper

Часові пояси.

```php
use Flowaxy\Support\Helpers\TimezoneHelper;

// Встановлення часового поясу
TimezoneHelper::set('Europe/Kyiv');

// Конвертація дати
$date = TimezoneHelper::convert($date, 'UTC', 'Europe/Kyiv');
```

### Робота з масивами

#### ArrayHelper

Робота з масивами.

```php
use Flowaxy\Support\Helpers\ArrayHelper;

// Отримання значення
$value = ArrayHelper::get($array, 'key.nested', 'default');

// Встановлення значення
ArrayHelper::set($array, 'key.nested', 'value');

// Перевірка наявності
if (ArrayHelper::has($array, 'key')) {
    // Ключ існує
}
```

### Робота з URL

#### UrlHelper

Робота з URL.

```php
use Flowaxy\Support\Helpers\UrlHelper;

// Генерація URL
$url = UrlHelper::to('page', ['id' => 123]);

// Перевірка URL
if (UrlHelper::is('admin/dashboard')) {
    // На сторінці dashboard
}
```

### Робота з сесіями

#### SessionHelper

Робота з сесіями.

```php
use Flowaxy\Support\Helpers\SessionHelper;

// Отримання значення
$value = SessionHelper::get('key', 'default');

// Встановлення значення
SessionHelper::put('key', 'value');

// Видалення
SessionHelper::forget('key');
```

### Робота з компонентами

#### ComponentHelper

Компоненти.

```php
use Flowaxy\Support\Helpers\ComponentHelper;

// Отримання компонента
$component = ComponentHelper::get('component-name', $config);
```

### Робота з маршрутами

#### RouteHelper

Маршрути.

```php
use Flowaxy\Support\Helpers\RouteHelper;

// Генерація URL маршруту
$url = RouteHelper::url('route-name', ['id' => 123]);
```

### Робота з запитами

#### RequestHelper

HTTP запити.

```php
use Flowaxy\Support\Helpers\RequestHelper;

// Отримання значення
$value = RequestHelper::get('key', 'default');
$value = RequestHelper::post('key', 'default');
$value = RequestHelper::input('key', 'default');

// Перевірка наявності
if (RequestHelper::has('key')) {
    // Ключ існує
}
```

### Робота з відповідями

#### ResponseHelper

HTTP відповіді.

```php
use Flowaxy\Support\Helpers\ResponseHelper;

// JSON відповідь
ResponseHelper::json(['data' => $data]);

// Редирект
ResponseHelper::redirect('/page');
```

### Робота з зображеннями

#### ImageHelper

Зображення.

```php
use Flowaxy\Support\Helpers\ImageHelper;

// Зміна розміру
ImageHelper::resize('path/to/image.jpg', 800, 600);

// Створення thumbnail
ImageHelper::thumbnail('path/to/image.jpg', 200, 200);
```

### Робота з завантаженням

#### UploadHelper

Завантаження файлів.

```php
use Flowaxy\Support\Helpers\UploadHelper;

// Завантаження файлу
$file = UploadHelper::file('file_input');

// Валідація та збереження
if ($file->isValid()) {
    $path = $file->store('uploads');
}
```

### Робота з архівами

#### ArchiveHelper

Архіви.

```php
use Flowaxy\Support\Helpers\ArchiveHelper;

// Створення архіву
ArchiveHelper::create('archive.zip', ['file1.txt', 'file2.txt']);

// Розпакування
ArchiveHelper::extract('archive.zip', 'destination/');
```

### Робота з CSV

#### CsvHelper

CSV файли.

```php
use Flowaxy\Support\Helpers\CsvHelper;

// Читання CSV
$data = CsvHelper::readFile('path/to/file.csv');

// Запис CSV
CsvHelper::writeFile('path/to/file.csv', $data);
```

### Робота з XML

#### XmlHelper

XML файли.

```php
use Flowaxy\Support\Helpers\XmlHelper;

// Читання XML
$data = XmlHelper::readFile('path/to/file.xml');

// Запис XML
XmlHelper::writeFile('path/to/file.xml', $data);
```

### Робота з YAML

#### YamlHelper

YAML файли.

```php
use Flowaxy\Support\Helpers\YamlHelper;

// Читання YAML
$data = YamlHelper::readFile('path/to/file.yaml');

// Запис YAML
YamlHelper::writeFile('path/to/file.yaml', $data);
```

### Робота з ролями

#### RoleHelper

Ролі користувачів.

```php
use Flowaxy\Support\Helpers\RoleHelper;

// Перевірка ролі
if (RoleHelper::has('admin')) {
    // Користувач адміністратор
}
```

### Робота з менеджерами

#### ManagerHelper

Менеджери системи.

```php
use Flowaxy\Support\Helpers\ManagerHelper;

// Отримання менеджера тем
$themeManager = ManagerHelper::themeManager();

// Отримання менеджера плагінів
$pluginManager = ManagerHelper::pluginManager();
```

### Робота з помилками

#### ErrorHelper

Обробка помилок.

```php
use Flowaxy\Support\Helpers\ErrorHelper;

// Логування помилки
ErrorHelper::log($exception);

// Відображення помилки
ErrorHelper::display($exception);
```

#### ExceptionHandler

Обробка винятків.

```php
use Flowaxy\Support\Helpers\ExceptionHandler;

// Обробка винятку
ExceptionHandler::handle($exception);
```

### Робота з ресурсами

#### AssetHelper

Ассети.

```php
use Flowaxy\Support\Helpers\AssetHelper;

// Підключення CSS
AssetHelper::css('path/to/style.css');

// Підключення JS
AssetHelper::js('path/to/script.js');
```

### Робота з контентом

#### ContentHelper

Контент.

```php
// Глобальні функції
render_component('component-name', $config);
render_template('template-name', $data);
get_content_data('hook-name', $default);
```

### Робота з API

#### ApiHelper

API endpoints.

```php
use Flowaxy\Support\Helpers\ApiHelper;

// Реєстрація API endpoint
ApiHelper::register('GET', '/api/users', function() {
    return ['users' => []];
});
```

## Глобальні функції

### Робота з темами

```php
// Компонент теми
theme_component('slider', $props);

// Asset теми
theme_asset('css/style.css', $versioning);

// Віджет теми
theme_widget('marketplace', $data);
```

### Робота з адмінкою

```php
// URL адмінки
admin_url('page');

// Редирект в адмінці
admin_redirect('page');

// Рендеринг layout
render_admin_layout($config);
```

### Робота з хуками

```php
// Actions
add_action('hook_name', $callback, $priority);
do_action('hook_name', ...$args);
has_action('hook_name');

// Filters
add_filter('hook_name', $callback, $priority);
apply_filters('hook_name', $value, ...$args);
has_filter('hook_name');
```

### Робота з безпекою

```php
// CSRF токен
csrf_token();

// Перевірка CSRF
verify_csrf_token($token);

// Санитизація
sanitize_input($input);
```

### Робота з шляхами

```php
// Шляхи
path_root();
path_flowaxy();
path_content();
path_storage();
path_uploads();
```

## Наступні кроки

- [Фасади](Facades) — всі фасади системи
- [Hooks API](Hooks-API) — API для роботи з хуками
- [Theme API](Theme-API) — API для роботи з темами
- [Plugin API](Plugin-API) — API для роботи з плагінами

---

**Хелпери забезпечують зручність та продуктивність!** 🛠️
