# Фасади

Повний довідник всіх 36 фасадів FLOWAXY-CMS.

## Огляд

Фасади надають статичний інтерфейс для доступу до сервісів, зареєстрованих в DI контейнері. Всі фасади успадковують `Flowaxy\Support\Facades\Facade`.

## Основні фасади

### App

Доступ до контейнера та конфігурації.

```php
use Flowaxy\Support\Facades\App;

// Отримання контейнера
$container = App::container();

// Створення об'єкта через контейнер
$service = App::make(ServiceInterface::class);

// Перевірка наявності
if (App::has(ServiceInterface::class)) {
    $service = App::make(ServiceInterface::class);
}

// Отримання конфігурації
$value = App::config('key', 'default');
```

### Cache

Робота з кешем.

```php
use Flowaxy\Support\Facades\Cache;

// Отримання значення
$value = Cache::get('key', 'default');

// Збереження значення
Cache::put('key', 'value', 3600); // TTL в секундах
Cache::set('key', 'value', 3600);

// Видалення
Cache::forget('key');
Cache::delete('key');

// Очищення всього кешу
Cache::clear();

// Збереження з callback (якщо не існує)
$value = Cache::remember('key', function() {
    return expensive_operation();
}, 3600);
```

### Database

Робота з базою даних.

```php
use Flowaxy\Support\Facades\Database;

// Отримання підключення
$pdo = Database::connection();

// Почати запит з таблиці
$users = Database::table('users')
    ->where('status', 'active')
    ->get();

// SELECT запит
$users = Database::select('SELECT * FROM users WHERE status = ?', ['active']);

// INSERT
Database::insert('users', [
    'name' => 'John',
    'email' => 'john@example.com',
]);

// UPDATE
Database::update('users', ['name' => 'Jane'], ['id' => 1]);

// DELETE
Database::delete('users', ['id' => 1]);
```

### Log / Logger

Логування.

```php
use Flowaxy\Support\Facades\Log;

// Різні рівні логування
Log::debug('Debug message');
Log::info('Info message');
Log::warning('Warning message');
Log::error('Error message');
Log::critical('Critical message');

// З контекстом
Log::info('User logged in', ['user_id' => 123]);
```

### Session

Управління сесіями.

```php
use Flowaxy\Support\Facades\Session;

// Отримання значення
$value = Session::get('key', 'default');

// Встановлення значення
Session::put('key', 'value');
Session::set('key', 'value');

// Видалення
Session::forget('key');
Session::remove('key');

// Очищення всієї сесії
Session::flush();
```

### Security

Безпека (CSRF, санитизація, валідація).

```php
use Flowaxy\Support\Facades\Security;

// CSRF токен
$token = Security::csrfToken();

// Перевірка CSRF
if (Security::verifyCsrfToken($token)) {
    // Безпечно
}

// Санитизація
$clean = Security::sanitize($input);

// Валідація
if (Security::validate($data, $rules)) {
    // Валідно
}
```

### Router

Маршрутизація.

```php
use Flowaxy\Support\Facades\Router;

// Отримання поточного маршруту
$route = Router::current();

// Генерація URL
$url = Router::url('page', ['id' => 123]);

// Перевірка активного маршруту
if (Router::is('admin/dashboard')) {
    // На сторінці dashboard
}
```

### Hooks

Хуки (Actions та Filters).

```php
use Flowaxy\Support\Facades\Hooks;

// Реєстрація Action
Hooks::on('init', function() {
    // Логіка
});

// Виконання Action
Hooks::dispatch('init');

// Реєстрація Filter
Hooks::filter('the_title', function($title) {
    return strtoupper($title);
});

// Застосування Filter
$title = Hooks::apply('the_title', 'Hello');
```

### Request

HTTP запити.

```php
use Flowaxy\Support\Facades\Request;

// Отримання значення
$value = Request::get('key', 'default');
$value = Request::post('key', 'default');
$value = Request::input('key', 'default');

// Перевірка наявності
if (Request::has('key')) {
    // Ключ існує
}

// Отримання всіх даних
$all = Request::all();

// Метод запиту
$method = Request::method(); // GET, POST, etc.

// URL
$url = Request::url();
$path = Request::path();
```

## Безпека

### Hash

Хешування паролів.

```php
use Flowaxy\Support\Facades\Hash;

// Хешування пароля
$hash = Hash::make('password');

// Перевірка пароля
if (Hash::check('password', $hash)) {
    // Пароль правильний
}
```

### Encryption

Шифрування даних.

```php
use Flowaxy\Support\Facades\Encryption;

// Шифрування
$encrypted = Encryption::encrypt('sensitive data');

// Розшифрування
$decrypted = Encryption::decrypt($encrypted);
```

### RateLimiter

Обмеження частоти запитів.

```php
use Flowaxy\Support\Facades\RateLimiter;

// Перевірка ліміту
if (RateLimiter::tooManyAttempts('key', 5)) {
    // Забагато спроб
}

// Збільшення лічильника
RateLimiter::hit('key', 60); // 60 секунд

// Очищення
RateLimiter::clear('key');
```

### Validator

Валідація даних.

```php
use Flowaxy\Support\Facades\Validator;

// Валідація
$validator = Validator::make($data, [
    'email' => 'required|email',
    'password' => 'required|min:8',
]);

if ($validator->fails()) {
    $errors = $validator->errors();
}
```

## Файли та медіа

### Storage

Файлова система.

```php
use Flowaxy\Support\Facades\Storage;

// Читання файлу
$content = Storage::get('path/to/file.txt');

// Запис файлу
Storage::put('path/to/file.txt', 'content');

// Перевірка існування
if (Storage::exists('path/to/file.txt')) {
    // Файл існує
}

// Видалення
Storage::delete('path/to/file.txt');
```

### Upload

Завантаження файлів.

```php
use Flowaxy\Support\Facades\Upload;

// Завантаження файлу
$file = Upload::file('file_input');

// Валідація та збереження
if ($file->isValid()) {
    $path = $file->store('uploads');
}
```

### Image

Робота з зображеннями.

```php
use Flowaxy\Support\Facades\Image;

// Відкриття зображення
$image = Image::open('path/to/image.jpg');

// Зміна розміру
$image->resize(800, 600);

// Збереження
$image->save('path/to/resized.jpg');
```

### Archive

Робота з архівами.

```php
use Flowaxy\Support\Facades\Archive;

// Створення архіву
Archive::create('archive.zip', ['file1.txt', 'file2.txt']);

// Розпакування
Archive::extract('archive.zip', 'destination/');
```

## Конфігурація та налаштування

### Config

Конфігурація.

```php
use Flowaxy\Support\Facades\Config;

// Отримання значення
$value = Config::get('key', 'default');

// Встановлення значення
Config::set('key', 'value');

// Перевірка наявності
if (Config::has('key')) {
    // Ключ існує
}
```

### Settings

Налаштування додатку.

```php
use Flowaxy\Support\Facades\Settings;

// Отримання налаштування
$value = Settings::get('setting_key', 'default');

// Встановлення налаштування
Settings::set('setting_key', 'value');
```

### Timezone

Часові пояси.

```php
use Flowaxy\Support\Facades\Timezone;

// Встановлення часового поясу
Timezone::set('Europe/Kyiv');

// Отримання поточного часового поясу
$timezone = Timezone::get();

// Конвертація дати
$date = Timezone::convert($date, 'UTC', 'Europe/Kyiv');
```

## Додаткові фасади

### Mail

Відправка пошти.

```php
use Flowaxy\Support\Facades\Mail;

// Відправка листа
Mail::send('template', ['data' => $data], function($message) {
    $message->to('user@example.com')
            ->subject('Subject');
});
```

### QueryBuilder

Побудова запитів.

```php
use Flowaxy\Support\Facades\QueryBuilder;

$query = QueryBuilder::table('users')
    ->select('name', 'email')
    ->where('status', 'active')
    ->orderBy('name')
    ->get();
```

### Asset

Управління ассетами.

```php
use Flowaxy\Support\Facades\Asset;

// Підключення CSS
Asset::css('path/to/style.css');

// Підключення JS
Asset::js('path/to/script.js');
```

### FeatureFlag

Feature flags.

```php
use Flowaxy\Support\Facades\FeatureFlag;

// Перевірка feature flag
if (FeatureFlag::enabled('new_feature')) {
    // Нова функціональність активна
}
```

### Role

Ролі користувачів.

```php
use Flowaxy\Support\Facades\Role;

// Перевірка ролі
if (Role::has('admin')) {
    // Користувач адміністратор
}

// Отримання ролей
$roles = Role::all();
```

### Theme

Темі.

```php
use Flowaxy\Support\Facades\Theme;

// Отримання активної теми
$theme = Theme::active();

// Рендеринг компонента
Theme::component('slider');

// Отримання asset
$url = Theme::asset('css/style.css');
```

### Plugin

Плагіни.

```php
use Flowaxy\Support\Facades\Plugin;

// Отримання плагіна
$plugin = Plugin::get('plugin-slug');

// Перевірка активності
if (Plugin::isActive('plugin-slug')) {
    // Плагін активний
}
```

## Робота з файлами

### Json

Робота з JSON.

```php
use Flowaxy\Support\Facades\Json;

// Читання JSON
$data = Json::read('path/to/file.json');

// Запис JSON
Json::write('path/to/file.json', $data);

// Декодування
$data = Json::decode($jsonString);

// Кодування
$json = Json::encode($data);
```

### Xml

Робота з XML.

```php
use Flowaxy\Support\Facades\Xml;

// Читання XML
$data = Xml::read('path/to/file.xml');

// Запис XML
Xml::write('path/to/file.xml', $data);
```

### Yaml

Робота з YAML.

```php
use Flowaxy\Support\Facades\Yaml;

// Читання YAML
$data = Yaml::read('path/to/file.yaml');

// Запис YAML
Yaml::write('path/to/file.yaml', $data);
```

### Csv

Робота з CSV.

```php
use Flowaxy\Support\Facades\Csv;

// Читання CSV
$data = Csv::read('path/to/file.csv');

// Запис CSV
Csv::write('path/to/file.csv', $data);
```

### Ini

Робота з INI.

```php
use Flowaxy\Support\Facades\Ini;

// Читання INI
$data = Ini::read('path/to/file.ini');

// Запис INI
Ini::write('path/to/file.ini', $data);
```

## Наступні кроки

- [Хелпери](Helpers) — всі хелпери системи
- [Hooks API](Hooks-API) — API для роботи з хуками
- [Theme API](Theme-API) — API для роботи з темами
- [Plugin API](Plugin-API) — API для роботи з плагінами
- [Security API](Security-API) — API безпеки

---

**Фасади забезпечують зручний доступ до функціоналу системи!** 🎯
