<?php

declare(strict_types=1);

namespace Flowaxy\Support\Traits;

use Flowaxy\Support\Actions\ActionHandler;
use Flowaxy\Support\Actions\ActionInterface;
use Flowaxy\Support\Helpers\SecurityHelper;
use Flowaxy\Support\Facades\Log;
use Exception;
use Throwable;

// Трейт для обработки действий
// Общие методы для обработки действий в страницах админки
trait ActionHandlerTrait
{
    // Получение экземпляра ActionHandler
    protected function getActionHandler(): ActionHandler
    {
        static $handler = null;
        if ($handler === null) {
            $handler = new ActionHandler();
        }
        return $handler;
    }

    // Регистрация действия
    protected function registerAction(ActionInterface $action): void
    {
        $this->getActionHandler()->register($action);
    }

    // Обработка действия
    protected function handleAction(string $actionName, array $params = [], bool $checkAccess = true): array
    {
        try {
            $handler = $this->getActionHandler();
            return $handler->execute($actionName, $params, $checkAccess);
        } catch (Throwable $e) {
            try {
                Log::Error("Помилка обробки дії '{$actionName}'", [
                    'error' => $e->getMessage(),
                    'exception' => $e,
                ]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    // Валидация действия
    protected function validateAction(string $actionName, array $params): array
    {
        $handler = $this->getActionHandler();
        $action = $handler->getAction($actionName);

        if (!$action) {
            return [
                'valid' => false,
                'error' => "Дія '{$actionName}' не знайдена",
            ];
        }

        return $action->validate($params);
    }

    // Выполнение действия
    protected function executeAction(string $actionName, array $params = []): array
    {
        return $this->handleAction($actionName, $params, true);
    }
}
