<?php

declare(strict_types=1);

namespace Tests\Infrastructure\Persistence\Database;

use Flowaxy\Infrastructure\Persistence\Database\QueryBuilder;
use PHPUnit\Framework\TestCase;

/**
 * Тесты для QueryBuilder
 *
 * @package Tests\Infrastructure\Persistence\Database
 */
final class QueryBuilderTest extends TestCase
{
    public function testSelect(): void
    {
        $query = new QueryBuilder();
        $query->select('id', 'name')->from('users');

        $sql = $query->toSql();
        $this->assertStringContainsString('SELECT', $sql);
        $this->assertStringContainsString('id', $sql);
        $this->assertStringContainsString('name', $sql);
        $this->assertStringContainsString('FROM users', $sql);
    }

    public function testWhere(): void
    {
        $query = new QueryBuilder();
        $query->select('*')->from('users')->where('id', '=', 1);

        $sql = $query->toSql();
        $this->assertStringContainsString('WHERE', $sql);
        $this->assertStringContainsString('id =', $sql);
    }

    public function testWhereIn(): void
    {
        $query = new QueryBuilder();
        $query->select('*')->from('users')->whereIn('id', [1, 2, 3]);

        $sql = $query->toSql();
        $this->assertStringContainsString('IN', $sql);
    }

    public function testOrderBy(): void
    {
        $query = new QueryBuilder();
        $query->select('*')->from('users')->orderBy('name', 'ASC');

        $sql = $query->toSql();
        $this->assertStringContainsString('ORDER BY', $sql);
        $this->assertStringContainsString('name ASC', $sql);
    }

    public function testLimit(): void
    {
        $query = new QueryBuilder();
        $query->select('*')->from('users')->limit(10);

        $sql = $query->toSql();
        $this->assertStringContainsString('LIMIT 10', $sql);
    }

    public function testRememberOnlyForSelect(): void
    {
        $query = new QueryBuilder();
        $query->select('*')->from('users')->remember('cache_key', 3600);

        // remember() должен работать только для SELECT
        $this->assertTrue(method_exists($query, 'remember'));
    }
}
