<?php

declare(strict_types=1);

namespace Tests\Contracts;

/**
 * Простой TestRunner для тестов контрактов
 * БЕЗ assertions DSL, lifecycle hooks, parallel execution
 *
 * @package Tests\Contracts
 */
final class TestRunner
{
    private array $results = [];
    private int $passed = 0;
    private int $failed = 0;

    /**
     * Запустить тест
     *
     * @param string $testName Название теста
     * @param callable $test Функция теста
     * @return void
     */
    public function run(string $testName, callable $test): void
    {
        try {
            $test();
            $this->results[] = ['name' => $testName, 'status' => 'passed'];
            $this->passed++;
        } catch (\Throwable $e) {
            $this->results[] = [
                'name' => $testName,
                'status' => 'failed',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ];
            $this->failed++;
        }
    }

    /**
     * Получить результаты
     *
     * @return array<string, mixed>
     */
    public function getResults(): array
    {
        return [
            'passed' => $this->passed,
            'failed' => $this->failed,
            'total' => $this->passed + $this->failed,
            'tests' => $this->results,
        ];
    }

    /**
     * Вывести результаты
     *
     * @return void
     */
    public function printResults(): void
    {
        echo "Contract Tests Results:\n";
        echo "Passed: {$this->passed}\n";
        echo "Failed: {$this->failed}\n";
        echo "Total: " . ($this->passed + $this->failed) . "\n\n";

        foreach ($this->results as $result) {
            if ($result['status'] === 'passed') {
                echo "✓ {$result['name']}\n";
            } else {
                echo "✗ {$result['name']}: {$result['error']}\n";
            }
        }
    }
}
