<?php

/**
 * Сервис для работы с темами
 * Инкапсулирует логику работы с темами с обработкой ошибок
 *
 * @package Flowaxy\Support\Services
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Services;

use Exception;
use Flowaxy\Core\System\PathResolver;
use Flowaxy\Support\Facades\Log;
use Flowaxy\Support\Helpers\DatabaseHelper;
use Flowaxy\Support\Helpers\FileHelper;
use Flowaxy\Support\Helpers\ManagerHelper;
use Flowaxy\Support\Helpers\ResourceCleanupHelper;
use Flowaxy\Support\Managers\ThemeManager;
use Throwable;

use function class_exists;
use function method_exists;
use const DS;

final class ThemeService
{
    /**
     * Получение менеджера тем
     *
     * @return ThemeManager|null
     */
    private function getThemeManager(): ?ThemeManager
    {
        return ManagerHelper::themeManager();
    }

    /**
     * Активация темы
     *
     * @param string $themeSlug Slug темы
     * @return bool Успех операции
     * @throws Exception Если ThemeManager недоступен
     */
    public function activate(string $themeSlug): bool
    {
        $themeManager = $this->getThemeManager();
        if (!$themeManager) {
            throw new Exception('ThemeManager не доступний');
        }

        try {
            $result = $themeManager->activateTheme($themeSlug);
            if ($result) {
                Log::Info('Тему активовано', ['theme' => $themeSlug]);
            } else {
                try {
                    Log::Warning('Помилка активації теми', ['theme' => $themeSlug]);
                } catch (Throwable $logError) {
                    // Ignore logging errors
                }
            }
            return $result;
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка активації теми', ['exception' => $e, 'theme' => $themeSlug]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
            return false;
        }
    }

    /**
     * Деактивация темы
     *
     * @param string $themeSlug Slug темы
     * @return bool Успех операции
     * @throws Exception Если ThemeManager недоступен
     */
    public function deactivate(string $themeSlug): bool
    {
        $themeManager = $this->getThemeManager();
        if (!$themeManager) {
            throw new Exception('ThemeManager не доступний');
        }

        try {
            $result = $themeManager->deactivateTheme($themeSlug);
            if ($result) {
                Log::Info('Тему деактивовано', ['theme' => $themeSlug]);
            } else {
                try {
                    Log::Warning('Помилка деактивації теми', ['theme' => $themeSlug]);
                } catch (Throwable $logError) {
                    // Ignore logging errors
                }
            }
            return $result;
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка деактивації теми', ['exception' => $e, 'theme' => $themeSlug]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
            return false;
        }
    }

    /**
     * Удаление темы
     * Примечание: Удаление темы включает удаление файлов и настроек из БД
     *
     * @param string $themeSlug Slug темы
     * @return bool Успех операции
     * @throws Exception Если ThemeManager недоступен или тема активна
     */
    public function delete(string $themeSlug): bool
    {
        $themeManager = $this->getThemeManager();
        if (!$themeManager) {
            throw new Exception('ThemeManager не доступний');
        }

        // Проверяем, что тема не активна
        $activeTheme = $themeManager->getActiveTheme();
        if ($activeTheme && isset($activeTheme['slug']) && $activeTheme['slug'] === $themeSlug) {
            throw new Exception('Неможливо видалити активну тему. Спочатку деактивуйте її.');
        }

        try {
            // Получаем путь к теме
            $themePath = PathResolver::themes() . DS . $themeSlug;

            if (!FileHelper::isDirectory($themePath)) {
                try {
                    Log::Warning('Тема не знайдена', ['theme' => $themeSlug]);
                } catch (Throwable $logError) {
                    // Ignore logging errors
                }
                return false;
            }

            // Удаляем настройки темы из БД
            $db = DatabaseHelper::getConnection();
            if ($db) {
                try {
                    $stmt = $db->prepare('DELETE FROM theme_settings WHERE theme_slug = ?');
                    $stmt->execute([$themeSlug]);
                } catch (Exception $e) {
                    try {
                        Log::Warning('Помилка видалення налаштувань теми з БД', ['exception' => $e, 'theme' => $themeSlug]);
                    } catch (Throwable $logError) {
                        // Ignore logging errors
                    }
                }
            }

            // Удаляем директорию темы
            ResourceCleanupHelper::deleteDirectory($themePath);

            // Очищаем кеш
            $themeManager->clearThemeCache($themeSlug);

            Log::Info('Тему видалено', ['theme' => $themeSlug]);
            return true;
        } catch (Throwable $e) {
            try {
                Log::Error('Помилка видалення теми', ['exception' => $e, 'theme' => $themeSlug]);
            } catch (Throwable $logError) {
                // Ignore logging errors
            }
            return false;
        }
    }

    /**
     * Загрузка темы из ZIP архива
     * Примечание: Этот метод будет реализован через ArchiveExtractionService
     *
     * @param string $zipPath Путь к ZIP архиву
     * @return array{success: bool, theme_slug?: string, error?: string}
     */
    public function upload(string $zipPath): array
    {
        // Этот метод будет делегирован ArchiveExtractionService
        // Пока возвращаем заглушку
        return [
            'success' => false,
            'error' => 'Метод upload() буде реалізовано через ArchiveExtractionService',
        ];
    }


    /**
     * Получение активной темы
     *
     * @return array<string, mixed>|null
     */
    public function getActiveTheme(): ?array
    {
        $themeManager = $this->getThemeManager();
        if (!$themeManager) {
            return null;
        }

        return $themeManager->getActiveTheme();
    }

    /**
     * Получение темы по slug
     *
     * @param string $themeSlug Slug темы
     * @return array<string, mixed>|null
     */
    public function getTheme(string $themeSlug): ?array
    {
        $themeManager = $this->getThemeManager();
        if (!$themeManager) {
            return null;
        }

        return $themeManager->getTheme($themeSlug);
    }
}
