<?php

/**
 * Фабрика для створення менеджерів сховища
 * Уніфікований доступ до різних типів сховища
 *
 * @package Flowaxy\Support\Managers
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Managers;

use Flowaxy\Contracts\Filesystem\StorageInterface;

class StorageFactory
{
    public const TYPE_COOKIE = 'cookie';
    public const TYPE_SESSION = 'session';
    public const TYPE_LOCAL_STORAGE = 'localStorage';
    public const TYPE_SESSION_STORAGE = 'sessionStorage';

    /**
     * Отримання менеджера сховища за типом
     *
     * @param string $type Тип сховища (cookie, session, localStorage, sessionStorage)
     * @param string $prefix Префікс для ключів (опціонально)
     * @return StorageInterface|null
     */
    public static function get(string $type, string $prefix = ''): ?StorageInterface
    {
        return match ($type) {
            self::TYPE_COOKIE => (function() use ($prefix) {
                $manager = CookieManager::getInstance();
                if ($prefix) {
                    $manager->setDefaultOptions(['path' => $prefix]);
                }
                return $manager;
            })(),
            self::TYPE_SESSION => (function() use ($prefix) {
                $manager = SessionManager::getInstance();
                if ($prefix) {
                    $manager->setPrefix($prefix);
                }
                return $manager;
            })(),
            self::TYPE_LOCAL_STORAGE => (function() use ($prefix) {
                $manager = StorageManager::getInstance();
                $manager->setType('localStorage');
                if ($prefix) {
                    $manager->setPrefix($prefix);
                }
                return $manager;
            })(),
            self::TYPE_SESSION_STORAGE => (function() use ($prefix) {
                $manager = StorageManager::getInstance();
                $manager->setType('sessionStorage');
                if ($prefix) {
                    $manager->setPrefix($prefix);
                }
                return $manager;
            })(),
            default => null,
        };
    }

    /**
     * Отримання менеджера cookies
     *
     * @param string $prefix Префікс для ключів (опціонально)
     * @return CookieManager
     */
    public static function cookie(string $prefix = ''): CookieManager
    {
        return self::get(self::TYPE_COOKIE, $prefix) ?? CookieManager::getInstance();
    }

    /**
     * Отримання менеджера сесій
     *
     * @param string $prefix Префікс для ключів (опціонально)
     * @return SessionManager
     */
    public static function session(string $prefix = ''): SessionManager
    {
        $manager = SessionManager::getInstance();
        if ($prefix) {
            $manager->setPrefix($prefix);
        }

        return $manager;
    }

    /**
     * Отримання менеджера localStorage
     *
     * @param string $prefix Префікс для ключів (опціонально)
     * @return StorageManager
     */
    public static function localStorage(string $prefix = ''): StorageManager
    {
        $manager = StorageManager::getInstance();
        $manager->setType('localStorage');
        if ($prefix) {
            $manager->setPrefix($prefix);
        }

        return $manager;
    }

    /**
     * Получение менеджера sessionStorage
     *
     * @param string $prefix Префикс для ключей (опционально)
     * @return StorageManager
     */
    public static function sessionStorage(string $prefix = ''): StorageManager
    {
        $manager = StorageManager::getInstance();
        $manager->setType('sessionStorage');
        if ($prefix) {
            $manager->setPrefix($prefix);
        }

        return $manager;
    }
}
