<?php

/**
 * Глобальные функции-хелперы для регистрации маршрутов
 * Используются плагинами и темами для регистрации публичных маршрутов
 *
 * @package Flowaxy\Support\Helpers
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

// Функции должны быть глобальными, без namespace

/**
 * Реєстрація публічного маршруту (front-end)
 *
 * Плагіни та теми НЕ повинні викликати $router->add() напряму.
 * Замість цього вони реєструють маршрути через цей хелпер, який підключається
 * до системного хука `public_routes`.
 *
 * @param string|array<string> $methods HTTP методи (GET, POST тощо)
 * @param string $path Шлях без початкового слеша (наприклад 'test-seo-page')
 * @param callable $handler Обробник маршруту
 * @param array<string,mixed> $options Додаткові опції для Router::add()
 * @return void
 */
if (!function_exists('register_public_route')) {
    function register_public_route(string|array $methods, string $path, callable $handler, array $options = []): void
    {
        // Використовуємо глобальний HookManager через hooks(), щоб не створювати дубль інстансу
        // Якщо доступний глобальний hooks() — використовуємо його
        if (function_exists('hooks')) {
            $methodsList = is_array($methods) ? $methods : [$methods];

            // Підписуємося на хук public_routes для публічних маршрутів
            hooks()->on('public_routes', function (\Flowaxy\Core\Routing\Router $router) use ($methodsList, $path, $handler, $options): void {
                $router->add($methodsList, $path, $handler, $options);
            });
            return;
        }

        // Fallback: якщо hooks() недоступний, пробуемо використати Hooks facade
        if (class_exists('\Flowaxy\Support\Facades\Hooks')) {
            try {
                \Flowaxy\Support\Facades\Hooks::dispatch('public_routes', $methods, $path, $handler, $options);
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }
    }
}
