<?php

/**
 * Хелпер для роботи з HTTP запитами
 * Обгортка над Request класом
 *
 * @package Flowaxy\Support\Helpers
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

use Flowaxy\Interface\Http\Request;

final class RequestHelper
{
    /**
     * Отримати значення з GET
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function get(string $key, mixed $default = null): mixed
    {
        return Request::getInstance()->get($key, $default);
    }

    /**
     * Отримати значення з POST
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function post(string $key, mixed $default = null): mixed
    {
        return Request::getInstance()->postValue($key, $default);
    }

    /**
     * Отримати значення з запиту (GET або POST)
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    public static function input(string $key, mixed $default = null): mixed
    {
        return Request::getInstance()->get($key, $default);
    }

    /**
     * Перевірити наявність ключа
     *
     * @param string $key
     * @return bool
     */
    public static function has(string $key): bool
    {
        return Request::getInstance()->has($key);
    }

    /**
     * Отримати всі дані запиту
     *
     * @return array<string, mixed>
     */
    public static function all(): array
    {
        return Request::getInstance()->all();
    }

    /**
     * Отримати тільки вказані ключі
     *
     * @param array<int, string> $keys
     * @return array<string, mixed>
     */
    public static function only(array $keys): array
    {
        return Request::getInstance()->only($keys);
    }

    /**
     * Отримати всі дані крім вказаних ключів
     *
     * @param array<int, string> $keys
     * @return array<string, mixed>
     */
    public static function except(array $keys): array
    {
        return Request::getInstance()->except($keys);
    }

    /**
     * Об'єднати дані з запитом
     *
     * @param array<string, mixed> $data
     * @return array<string, mixed>
     */
    public static function merge(array $data): array
    {
        return array_merge(self::all(), $data);
    }

    /**
     * Перевірити метод запиту
     *
     * @param string $method
     * @return bool
     */
    public static function isMethod(string $method): bool
    {
        return Request::getInstance()->isMethod($method);
    }

    /**
     * Перевірити, чи це AJAX запит
     *
     * @return bool
     */
    public static function isAjax(): bool
    {
        return Request::getInstance()->isAjax();
    }

    /**
     * Перевірити, чи це JSON запит
     *
     * @return bool
     */
    public static function isJson(): bool
    {
        return Request::getInstance()->wantsJson();
    }

    /**
     * Перевірити, чи очікується JSON відповідь
     *
     * @return bool
     */
    public static function wantsJson(): bool
    {
        return Request::getInstance()->wantsJson();
    }

    /**
     * Отримати IP адресу клієнта
     *
     * @return string
     */
    public static function ip(): string
    {
        return Request::getInstance()->ip();
    }

    /**
     * Отримати User-Agent
     *
     * @return string
     */
    public static function userAgent(): string
    {
        return \Flowaxy\Infrastructure\Security\RequestFilter::server('HTTP_USER_AGENT', '', 'string');
    }

    /**
     * Отримати URL запиту
     *
     * @return string
     */
    public static function url(): string
    {
        return Request::getInstance()->url();
    }

    /**
     * Отримати повний URL запиту
     *
     * @return string
     */
    public static function fullUrl(): string
    {
        return Request::getInstance()->fullUrl();
    }

    /**
     * Отримати шлях запиту
     *
     * @return string
     */
    public static function path(): string
    {
        return Request::getInstance()->path();
    }

    /**
     * Отримати метод запиту
     *
     * @return string
     */
    public static function method(): string
    {
        return Request::getInstance()->method();
    }

    /**
     * Перевірити, чи використовується HTTPS
     *
     * @return bool
     */
    public static function isSecure(): bool
    {
        return Request::getInstance()->isSecure();
    }

    /**
     * Отримати заголовок
     *
     * @param string $name
     * @param string|null $default
     * @return string|null
     */
    public static function header(string $name, ?string $default = null): ?string
    {
        $key = 'HTTP_' . str_replace('-', '_', strtoupper($name));

        return \Flowaxy\Infrastructure\Security\RequestFilter::server($key, $default);
    }

    /**
     * Отримати cookie
     *
     * @param string $name
     * @param mixed $default
     * @return mixed
     */
    public static function cookie(string $name, mixed $default = null): mixed
    {
        return \Flowaxy\Infrastructure\Security\RequestFilter::cookie($name, $default);
    }

    /**
     * Отримати файл
     *
     * @param string $key
     * @return array<string, mixed>|null
     */
    public static function file(string $key): ?array
    {
        return $_FILES[$key] ?? null;
    }

    /**
     * Отримати Bearer токен
     *
     * @return string|null
     */
    public static function bearerToken(): ?string
    {
        $header = self::header('Authorization');
        if ($header === null) {
            return null;
        }

        if (str_starts_with($header, 'Bearer ')) {
            return substr($header, 7);
        }

        return null;
    }
}
