<?php

/**
 * Хелпер для работы с уведомлениями из массивов конфигурации
 *
 * @package Flowaxy\Support\Helpers
 */

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

final class NotificationHelper
{
    /**
     * Рендеринг уведомления из массива конфигурации
     *
     * @param array<string, mixed> $config Конфигурация уведомления
     * @return string HTML уведомления
     */
    public static function render(array $config): string
    {
        $message = $config['message'] ?? '';
        $type = $config['type'] ?? 'info'; // success, danger, warning, info
        $dismissible = $config['dismissible'] ?? true;
        $icon = $config['icon'] ?? null;

        $alertClass = 'alert alert-' . htmlspecialchars($type);
        if ($dismissible) {
            $alertClass .= ' alert-dismissible fade show';
        }

        $iconHtml = '';
        if ($icon !== null) {
            $iconClass = str_starts_with($icon, 'fa-') ? 'fas ' . $icon : 'fas fa-' . $icon;
            $iconHtml = '<i class="' . htmlspecialchars($iconClass) . ' me-2"></i>';
        }

        $html = '<div class="' . $alertClass . '" role="alert">';
        $html .= $iconHtml . htmlspecialchars($message);

        if ($dismissible) {
            $html .= '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>';
        }

        $html .= '</div>';

        return $html;
    }

    /**
     * Отображение уведомления (с автоскрытием)
     *
     * @param string $message Сообщение
     * @param string $type Тип уведомления
     * @param int $delay Задержка перед автоскрытием (мс, 0 = без автоскрытия)
     * @return string HTML уведомления с JavaScript
     */
    public static function show(string $message, string $type = 'info', int $delay = 5000): string
    {
        $config = [
            'message' => $message,
            'type' => $type,
            'dismissible' => true,
        ];

        $html = self::render($config);

        if ($delay > 0) {
            $html .= '<script>
                setTimeout(function() {
                    const alert = document.querySelector(".alert");
                    if (alert) {
                        const bsAlert = new bootstrap.Alert(alert);
                        bsAlert.close();
                    }
                }, ' . $delay . ');
            </script>';
        }

        return $html;
    }
}
