<?php

/**
 * Хелпер для работы с ассетами модулей
 *
 * @package Flowaxy\Support\Helpers
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

use Flowaxy\Core\System\PathResolver;

final class ModuleAssetHelper
{
    /**
     * Получить URL ассета модуля
     *
     * @param string $moduleName Имя модуля (например: Installer, Admin, API)
     * @param string $path Относительный путь к файлу (например: css/style.css)
     * @param bool $versioning Добавить версию для cache busting
     * @return string
     */
    public static function url(string $moduleName, string $path, bool $versioning = true): string
    {
        $baseUrl = self::getBaseUrl();
        $assetPath = ltrim($path, '/');
        $url = $baseUrl . '/assets/modules/' . strtolower($moduleName) . '/' . $assetPath;

        if ($versioning) {
            $filePath = self::getFilePath($moduleName, $assetPath);
            if ($filePath && file_exists($filePath)) {
                $version = filemtime($filePath);
                $url .= '?v=' . $version;
            }
        }

        return $url;
    }

    /**
     * Получить путь к файлу ассета модуля
     *
     * @param string $moduleName Имя модуля
     * @param string $path Относительный путь к файлу
     * @return string|null
     */
    public static function getFilePath(string $moduleName, string $path): ?string
    {
        $root = PathResolver::root();
        $assetPath = ltrim($path, '/');
        $filePath = $root . DS . 'assets' . DS . 'modules' . DS . strtolower($moduleName) . DS . str_replace('/', DS, $assetPath);

        return file_exists($filePath) ? $filePath : null;
    }

    /**
     * Получить URL CSS файла модуля
     *
     * @param string $moduleName Имя модуля
     * @param string $file Имя файла (например: style.css)
     * @param bool $versioning
     * @return string
     */
    public static function css(string $moduleName, string $file, bool $versioning = true): string
    {
        return self::url($moduleName, 'css/' . ltrim($file, '/'), $versioning);
    }

    /**
     * Получить URL JS файла модуля
     *
     * @param string $moduleName Имя модуля
     * @param string $file Имя файла (например: script.js)
     * @param bool $versioning
     * @return string
     */
    public static function js(string $moduleName, string $file, bool $versioning = true): string
    {
        return self::url($moduleName, 'scripts/' . ltrim($file, '/'), $versioning);
    }

    /**
     * Получить URL изображения модуля
     *
     * @param string $moduleName Имя модуля
     * @param string $file Имя файла (например: logo.png)
     * @param bool $versioning
     * @return string
     */
    public static function image(string $moduleName, string $file, bool $versioning = true): string
    {
        return self::url($moduleName, 'images/' . ltrim($file, '/'), $versioning);
    }

    /**
     * Получить базовый URL
     *
     * @return string
     */
    private static function getBaseUrl(): string
    {
        // Определяем базовый URL из конфигурации или из окружения
        if (defined('SITE_URL')) {
            return rtrim(\SITE_URL, '/');
        }

        // Fallback: определяем из текущего запроса
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $scriptName = dirname($_SERVER['SCRIPT_NAME'] ?? '');

        return $protocol . '://' . $host . rtrim($scriptName, '/');
    }

    /**
     * Получить путь к SCSS исходникам модуля
     *
     * @param string $moduleName Имя модуля
     * @return string
     */
    public static function getScssPath(string $moduleName): string
    {
        $root = PathResolver::root();
        return $root . DS . 'assets' . DS . 'modules' . DS . strtolower($moduleName) . DS . 'scss';
    }

    /**
     * Получить путь к скомпилированным CSS файлам модуля
     *
     * @param string $moduleName Имя модуля
     * @return string
     */
    public static function getCssPath(string $moduleName): string
    {
        $root = PathResolver::root();
        return $root . DS . 'assets' . DS . 'modules' . DS . strtolower($moduleName) . DS . 'css';
    }
}
