<?php

/**
 * Хелпер для работы с модальными окнами из массивов конфигурации
 *
 * @package Flowaxy\Support\Helpers
 */

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

final class ModalHelper
{
    /**
     * Рендеринг модального окна из массива конфигурации
     *
     * @param array<string, mixed> $config Конфигурация модального окна
     * @return string HTML модального окна
     */
    public static function render(array $config): string
    {
        $id = $config['id'] ?? 'modal_' . uniqid();
        $title = $config['title'] ?? 'Модальное окно';
        $body = $config['body'] ?? '';
        $footer = $config['footer'] ?? [];
        $size = $config['size'] ?? ''; // sm, lg, xl
        $backdrop = $config['backdrop'] ?? true;
        $keyboard = $config['keyboard'] ?? true;

        $modalClass = 'modal fade';
        $dialogClass = 'modal-dialog';
        if (!empty($size)) {
            $dialogClass .= ' modal-' . htmlspecialchars($size);
        }

        $dataAttributes = '';
        if (!$backdrop) {
            $dataAttributes .= ' data-bs-backdrop="static"';
        }
        if (!$keyboard) {
            $dataAttributes .= ' data-bs-keyboard="false"';
        }

        $html = '<div class="' . $modalClass . '" id="' . htmlspecialchars($id) . '" tabindex="-1"' . $dataAttributes . '>';
        $html .= '<div class="' . $dialogClass . '">';
        $html .= '<div class="modal-content">';

        // Header
        $html .= '<div class="modal-header">';
        $html .= '<h5 class="modal-title">' . htmlspecialchars($title) . '</h5>';
        $html .= '<button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>';
        $html .= '</div>';

        // Body
        $html .= '<div class="modal-body">';
        $html .= $body;
        $html .= '</div>';

        // Footer
        if (!empty($footer)) {
            $html .= '<div class="modal-footer">';
            foreach ($footer as $button) {
                if (is_array($button)) {
                    $html .= ButtonHelper::render($button);
                } else {
                    $html .= $button;
                }
            }
            $html .= '</div>';
        }

        $html .= '</div>';
        $html .= '</div>';
        $html .= '</div>';

        return $html;
    }

    /**
     * Открытие модального окна через JavaScript
     *
     * @param string $modalId ID модального окна
     * @return string JavaScript код
     */
    public static function open(string $modalId): string
    {
        return '<script>
            document.addEventListener("DOMContentLoaded", function() {
                const modalElement = document.getElementById("' . htmlspecialchars($modalId) . '");
                if (modalElement) {
                    const modal = new bootstrap.Modal(modalElement);
                    modal.show();
                }
            });
        </script>';
    }

    /**
     * Закрытие модального окна через JavaScript
     *
     * @param string $modalId ID модального окна
     * @return string JavaScript код
     */
    public static function close(string $modalId): string
    {
        return '<script>
            const modalElement = document.getElementById("' . htmlspecialchars($modalId) . '");
            if (modalElement) {
                const modal = bootstrap.Modal.getInstance(modalElement);
                if (modal) {
                    modal.hide();
                }
            }
        </script>';
    }

    /**
     * Загрузка контента в модальное окно через AJAX
     *
     * @param string $modalId ID модального окна
     * @param string $url URL для загрузки контента
     * @return string JavaScript код
     */
    public static function loadAjax(string $modalId, string $url): string
    {
        return '<script>
            document.addEventListener("DOMContentLoaded", function() {
                const modalElement = document.getElementById("' . htmlspecialchars($modalId) . '");
                if (modalElement) {
                    const modalBody = modalElement.querySelector(".modal-body");
                    if (modalBody) {
                        fetch("' . htmlspecialchars($url) . '")
                            .then(response => response.text())
                            .then(data => {
                                modalBody.innerHTML = data;
                                const modal = new bootstrap.Modal(modalElement);
                                modal.show();
                            })
                            .catch(error => {
                                console.error("Error loading modal content:", error);
                            });
                    }
                }
            });
        </script>';
    }
}
