<?php

/**
 * Хелпер для работы с меню из массивов конфигурации
 *
 * @package Flowaxy\Support\Helpers
 */

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

final class MenuHelper
{
    /**
     * Построение структуры меню из массива
     *
     * @param array<int, array<string, mixed>> $menuItems Массив пунктов меню
     * @param string $currentPage Текущая страница
     * @return string HTML меню
     */
    public static function render(array $menuItems, string $currentPage = ''): string
    {
        // Сортируем по order
        usort($menuItems, function($a, $b) {
            $orderA = $a['order'] ?? 999;
            $orderB = $b['order'] ?? 999;
            return $orderA <=> $orderB;
        });

        $html = '<ul class="nav flex-column wp-menu">';

        foreach ($menuItems as $item) {
            $html .= self::renderMenuItem($item, $currentPage);
        }

        $html .= '</ul>';

        return $html;
    }

    /**
     * Рендеринг одного пункта меню
     *
     * @param array<string, mixed> $item Пункт меню
     * @param string $currentPage Текущая страница
     * @return string HTML пункта меню
     */
    private static function renderMenuItem(array $item, string $currentPage): string
    {
        $text = $item['text'] ?? '';
        $href = $item['href'] ?? '#';
        $icon = $item['icon'] ?? '';
        $page = $item['page'] ?? '';
        $submenu = $item['submenu'] ?? [];
        $permission = $item['permission'] ?? null;

        // Проверка прав доступа
        if ($permission !== null && function_exists('current_user_can')) {
            if (!current_user_can($permission)) {
                return '';
            }
        }

        $isActive = ($page === $currentPage);
        $hasSubmenu = !empty($submenu);

        $iconHtml = '';
        if (!empty($icon)) {
            $iconClass = str_starts_with($icon, 'fa-') ? 'fas ' . $icon : 'fas fa-' . $icon;
            $iconHtml = '<i class="' . htmlspecialchars($iconClass) . '"></i> ';
        }

        $html = '<li class="nav-item' . ($isActive ? ' active' : '') . '">';

        if ($hasSubmenu) {
            $html .= '<a class="nav-link" href="' . htmlspecialchars($href) . '" data-bs-toggle="collapse" data-bs-target="#submenu-' . htmlspecialchars($page) . '">';
            $html .= $iconHtml . htmlspecialchars($text);
            $html .= '<i class="fas fa-chevron-down ms-auto"></i>';
            $html .= '</a>';
            $html .= '<div class="collapse' . ($isActive ? ' show' : '') . '" id="submenu-' . htmlspecialchars($page) . '">';
            $html .= '<ul class="nav flex-column ms-3">';
            foreach ($submenu as $subItem) {
                $html .= self::renderMenuItem($subItem, $currentPage);
            }
            $html .= '</ul>';
            $html .= '</div>';
        } else {
            $html .= '<a class="nav-link' . ($isActive ? ' active' : '') . '" href="' . htmlspecialchars($href) . '">';
            $html .= $iconHtml . htmlspecialchars($text);
            $html .= '</a>';
        }

        $html .= '</li>';

        return $html;
    }

    /**
     * Получение активных пунктов меню
     *
     * @param array<int, array<string, mixed>> $menuItems Массив пунктов меню
     * @param string $currentPage Текущая страница
     * @return array<int, array<string, mixed>> Активные пункты меню
     */
    public static function getActiveItems(array $menuItems, string $currentPage): array
    {
        $activeItems = [];

        foreach ($menuItems as $item) {
            $page = $item['page'] ?? '';
            if ($page === $currentPage) {
                $activeItems[] = $item;
            }

            $submenu = $item['submenu'] ?? [];
            if (!empty($submenu)) {
                $activeSubmenu = self::getActiveItems($submenu, $currentPage);
                if (!empty($activeSubmenu)) {
                    $activeItems = array_merge($activeItems, $activeSubmenu);
                }
            }
        }

        return $activeItems;
    }
}
