<?php

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

use Flowaxy\Infrastructure\Mail\Mail;

// Хелпер для роботи з поштою
// Обгортка над Mail класом
final class MailHelper
{
    // Відправити лист
    public static function send(string|array $to, string $subject, string $body, bool $isHtml = true): bool
    {
        $mail = new Mail();
        $mail->to($to)->subject($subject)->body($body, $isHtml);

        return $mail->send();
    }

    // Додати лист в чергу
    public static function queue(string|array $to, string $subject, string $body, bool $isHtml = true): bool
    {
        // Поки що просто відправляємо, в майбутньому можна додати чергу
        return self::send($to, $subject, $body, $isHtml);
    }

    // Встановити отримувача
    public static function to(Mail $mail, string|array $to): Mail
    {
        return $mail->to($to);
    }

    // Встановити відправника
    public static function from(Mail $mail, string $from, ?string $name = null): Mail
    {
        return $mail->from($from, $name);
    }

    // Встановити тему
    public static function subject(Mail $mail, string $subject): Mail
    {
        return $mail->subject($subject);
    }

    // Встановити тіло листа
    public static function body(Mail $mail, string $body, bool $isHtml = true): Mail
    {
        return $mail->body($body, $isHtml);
    }

    // Додати копію
    public static function cc(Mail $mail, string|array $cc): Mail
    {
        return $mail->cc($cc);
    }

    // Додати приховану копію
    public static function bcc(Mail $mail, string|array $bcc): Mail
    {
        return $mail->bcc($bcc);
    }

    // Встановити адресу для відповіді
    public static function replyTo(Mail $mail, string $replyTo): Mail
    {
        return $mail->header('Reply-To', $replyTo);
    }

    // Додати вкладення
    public static function attach(Mail $mail, string $path, ?string $name = null): Mail
    {
        return $mail->attach($path, $name);
    }

    // Вставити зображення
    // Прикріплює файл як вкладення (функціональність embed не реалізована в базовому Mail класі)
    public static function embed(Mail $mail, string $path, string $cid): Mail
    {
        // Базова реалізація - прикріплюємо як звичайний файл
        // В майбутньому можна розширити для вбудованих зображень
        return $mail->attach($path, $cid);
    }

    // Встановити пріоритет
    public static function priority(Mail $mail, int $priority): Mail
    {
        // Встановлюємо X-Priority заголовок
        return $mail->header('X-Priority', (string)$priority);
    }
}
