<?php

/**
 * Хелпер для работы с layout компонентами (Header, Footer, Sidebar) из массивов конфигурации
 *
 * @package Flowaxy\Support\Helpers
 */

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

final class LayoutHelper
{
    /**
     * Рендеринг Header из массива конфигурации
     *
     * @param array<string, mixed> $config Конфигурация Header
     * @return string HTML Header
     */
    public static function renderHeader(array $config = []): string
    {
        $title = $config['title'] ?? 'Админ-панель';
        $user = $config['user'] ?? null;
        $menuItems = $config['menu'] ?? [];

        $html = '<header class="admin-header">';
        $html .= '<div class="container-fluid">';
        $html .= '<div class="row align-items-center">';
        $html .= '<div class="col-md-6">';
        $html .= '<h1 class="h3 mb-0">' . htmlspecialchars($title) . '</h1>';
        $html .= '</div>';

        if ($user !== null) {
            $html .= '<div class="col-md-6 text-end">';
            $html .= '<span class="text-muted">' . htmlspecialchars($user['name'] ?? 'Пользователь') . '</span>';
            $html .= '</div>';
        }

        $html .= '</div>';
        $html .= '</div>';
        $html .= '</header>';

        return $html;
    }

    /**
     * Рендеринг Footer из массива конфигурации
     *
     * @param array<string, mixed> $config Конфигурация Footer
     * @return string HTML Footer
     */
    public static function renderFooter(array $config = []): string
    {
        $text = $config['text'] ?? 'Дякуємо за довіру до web студії <a href="https://flowaxy.com" target="_blank" rel="noopener noreferrer">FLOWAXY</a>';
        $version = $config['version'] ?? '1.0.0 Alpha prerelease';

        $html = '<footer class="admin-footer">';
        $html .= '<div class="container-fluid">';
        $html .= '<div class="row align-items-center">';
        $html .= '<div class="col-md-6">';
        $html .= '<p class="mb-0 text-muted">' . $text . '</p>';
        $html .= '</div>';
        $html .= '<div class="col-md-6 text-md-end">';
        $html .= '<p class="mb-0 text-muted">Версія ' . htmlspecialchars($version) . '</p>';
        $html .= '</div>';
        $html .= '</div>';
        $html .= '</div>';
        $html .= '</footer>';

        return $html;
    }

    /**
     * Рендеринг Sidebar из массива конфигурации
     *
     * @param array<string, mixed> $config Конфигурация Sidebar
     * @return string HTML Sidebar
     */
    public static function renderSidebar(array $config = []): string
    {
        $menuItems = $config['menu'] ?? [];
        $currentPage = $config['currentPage'] ?? '';

        $html = '<nav id="sidebarMenu" class="sidebar d-md-block show">';
        $html .= '<div class="sidebar-content">';
        $html .= MenuHelper::render($menuItems, $currentPage);
        $html .= '</div>';
        $html .= '</nav>';

        return $html;
    }

    /**
     * Рендеринг полного layout (Header + Sidebar + Content + Footer)
     *
     * @param array<string, mixed> $config Конфигурация layout
     * @return string HTML layout
     */
    public static function renderLayout(array $config): string
    {
        $header = $config['header'] ?? [];
        $sidebar = $config['sidebar'] ?? [];
        $content = $config['content'] ?? '';
        $footer = $config['footer'] ?? [];

        $html = '<div class="admin-layout">';
        $html .= self::renderHeader($header);
        $html .= '<div class="container-fluid">';
        $html .= '<div class="row">';
        $html .= '<div class="col-md-3">';
        $html .= self::renderSidebar($sidebar);
        $html .= '</div>';
        $html .= '<div class="col-md-9">';
        $html .= $content;
        $html .= '</div>';
        $html .= '</div>';
        $html .= '</div>';
        $html .= self::renderFooter($footer);
        $html .= '</div>';

        return $html;
    }
}
