<?php

/**
 * Глобальные функции-обертки для обработки ошибок
 * Используют модуль Errors для отображения страниц ошибок
 *
 * @package Flowaxy\Support\Helpers
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

// Функции должны быть глобальными, без namespace

/**
 * Відображення сторінки помилки за HTTP кодом
 *
 * @param int $httpCode HTTP код помилки (400, 404, 500, тощо)
 * @param string|null $customTitle Користувацький заголовок
 * @param string|null $customMessage Користувацьке повідомлення
 * @param array<string, mixed>|null $debugInfo Debug інформація (для 500+)
 * @return void
 */
if (!function_exists('showHttpError')) {
    function showHttpError(int $httpCode, ?string $customTitle = null, ?string $customMessage = null, ?array $debugInfo = null): void
    {
        // Используем модуль Errors (стандартный дизайн) для всех случаев - и админка, и публичная сторона
        $errorModule = null;

        // Сначала пробуем путь из flowaxy/Core/System/Errors/
        if (\class_exists('Flowaxy\Core\System\Errors\ErrorModuleLoader')) {
            try {
                $errorModule = \Flowaxy\Core\System\Errors\ErrorModuleLoader::load();
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        // Fallback: пробуем старый путь из modules/Errors/
        if ($errorModule === null && \class_exists('Flowaxy\Modules\Errors\ErrorModuleLoader')) {
            try {
                $errorModule = \Flowaxy\Modules\Errors\ErrorModuleLoader::load();
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        // Fallback: пробуем напрямую ErrorModule (если уже загружен)
        if ($errorModule === null && \class_exists('Flowaxy\Modules\Errors\ErrorModule')) {
            try {
                $errorModule = \Flowaxy\Modules\Errors\ErrorModule::getInstance();
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        if ($errorModule !== null) {
            try {
                $handler = $errorModule->getHandler();
                $handler->showHttpError($httpCode, $customTitle, $customMessage, $debugInfo);
                exit;
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        // Fallback - минимальное сообщение только если модуль Errors недоступен
        if (\headers_sent()) {
            return;
        }

        if (!\headers_sent()) {
            \http_response_code($httpCode);
            \header('Content-Type: text/html; charset=UTF-8');
        }

        echo 'Модуль Errors недоступний. HTTP код: ' . $httpCode;
        if ($customMessage) {
            echo ' - ' . \htmlspecialchars($customMessage, ENT_QUOTES, 'UTF-8');
        }
        exit;
    }
}

/**
 * Функція для відображення сторінки помилки 500 з детальною інформацією
 *
 * @param Throwable|null $exception Виняток, який викликав помилку
 * @param array<string, mixed>|null $errorInfo Додаткова інформація про помилку (file, line, message)
 * @return void
 */
if (!function_exists('showError500Page')) {
    function showError500Page(?\Throwable $exception = null, ?array $errorInfo = null): void
    {
        // Пробуем загрузить модуль Errors
        $errorModule = null;

        if (\class_exists('Flowaxy\Core\System\Errors\ErrorModuleLoader')) {
            try {
                $errorModule = \Flowaxy\Core\System\Errors\ErrorModuleLoader::load();
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        if ($errorModule === null && \class_exists('Flowaxy\Modules\Errors\ErrorModule')) {
            try {
                $errorModule = \Flowaxy\Modules\Errors\ErrorModule::getInstance();
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        if ($errorModule !== null) {
            try {
                $handler = $errorModule->getHandler();
                $handler->showError500Page($exception, $errorInfo);
                return;
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        // Fallback - минимальное сообщение, если модуль недоступен
        if (!\headers_sent()) {
            \http_response_code(500);
            \header('Content-Type: text/html; charset=UTF-8');
        }

        echo 'Модуль Errors недоступний. Внутрішня помилка сервера.';
        if ($errorInfo && isset($errorInfo['message'])) {
            echo ' ' . \htmlspecialchars($errorInfo['message'], ENT_QUOTES, 'UTF-8');
        } elseif ($exception) {
            echo ' ' . \htmlspecialchars($exception->getMessage(), ENT_QUOTES, 'UTF-8');
        }
        exit;
    }
}

/**
 * Перевірка стану системи (БД, таблиці) перед обробкою запитів
 * Якщо система не готова - показує 500 помилку
 *
 * @return void
 */
if (!function_exists('checkSystemState')) {
    function checkSystemState(): void
    {
        // Пробуем загрузить модуль Errors
        $errorModule = null;

        if (\class_exists('Flowaxy\Core\System\Errors\ErrorModuleLoader')) {
            try {
                $errorModule = \Flowaxy\Core\System\Errors\ErrorModuleLoader::load();
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        if ($errorModule === null && \class_exists('Flowaxy\Modules\Errors\ErrorModule')) {
            try {
                $errorModule = \Flowaxy\Modules\Errors\ErrorModule::getInstance();
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        if ($errorModule !== null) {
            try {
                $handler = $errorModule->getHandler();
                $handler->checkSystemState();
                return;
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        // Установщик удален - система работает без проверки installed.flag
        // Root пользователь позволяет войти в админку даже без БД
        // Файл installed.flag создается автоматически при первом запуске
    }
}

/**
 * Показує сторінку з повідомленням про необхідність установки системи
 *
 * @return void
 */
if (!function_exists('showInstallationRequired')) {
    function showInstallationRequired(): void
    {
        // Пробуем загрузить модуль Errors
        $errorModule = null;

        if (\class_exists('Flowaxy\Core\System\Errors\ErrorModuleLoader')) {
            try {
                $errorModule = \Flowaxy\Core\System\Errors\ErrorModuleLoader::load();
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        if ($errorModule === null && \class_exists('Flowaxy\Modules\Errors\ErrorModule')) {
            try {
                $errorModule = \Flowaxy\Modules\Errors\ErrorModule::getInstance();
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        if ($errorModule !== null) {
            try {
                $installationService = $errorModule->getInstallationService();
                $installationService->show();
                exit;
            } catch (\Throwable $e) {
                // Игнорируем ошибки
            }
        }

        // Fallback - минимальное сообщение
        if (!\headers_sent()) {
            \http_response_code(503);
            \header('Content-Type: text/html; charset=UTF-8');
        }
        echo '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Сервіс тимчасово недоступний</title></head><body>';
        echo '<h1>Сервіс тимчасово недоступний</h1>';
        echo '<p>Будь ласка, спробуйте пізніше.</p>';
        echo '</body></html>';
        exit;
    }
}
