<?php

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

use Flowaxy\Infrastructure\Config\SystemConfig;
use Flowaxy\Support\Facades\Settings;

use function class_exists;
use function method_exists;

// Хелпер для роботи з конфігурацією
// Обгортка над SystemConfig
final class ConfigHelper
{
    // Отримати значення конфігурації
    public static function get(string $key, mixed $default = null): mixed
    {
        return SystemConfig::getInstance()->get($key, $default);
    }

    // Встановити значення конфігурації
    public static function set(string $key, mixed $value): void
    {
        if (class_exists(Settings::class)) {
            Settings::set($key, $value);
        }
    }

    // Перевірити наявність ключа
    public static function has(string $key): bool
    {
        if (class_exists(Settings::class)) {
            return Settings::has($key);
        }

        return self::get($key) !== null;
    }

    // Отримати всі значення конфігурації
    public static function all(): array
    {
        if (class_exists(Settings::class)) {
            $manager = Settings::manager();
            if ($manager !== null) {
                return $manager->all();
            }
        }

        return [];
    }

    // Завантажити конфігурацію з файлу
    public static function load(string $path): array|false
    {
        $extension = FileHelper::extension($path);

        return match ($extension) {
            'php' => include $path,
            'json' => JsonHelper::readFile($path),
            'ini' => IniHelper::readFile($path),
            'yaml', 'yml' => YamlHelper::readFile($path),
            default => false,
        };
    }

    // Зберегти конфігурацію у файл
    public static function save(string $path, array $data): bool
    {
        $extension = FileHelper::extension($path);

        return match ($extension) {
            'json' => JsonHelper::writeFile($path, $data),
            'ini' => IniHelper::writeFile($path, $data),
            'yaml', 'yml' => YamlHelper::writeFile($path, $data),
            default => false,
        };
    }

    // Об'єднати конфігурації
    public static function merge(array $config1, array $config2): array
    {
        return array_merge_recursive($config1, $config2);
    }

    // Видалити ключ з конфігурації
    public static function forget(string $key): void
    {
        if (class_exists(Settings::class)) {
            $manager = Settings::manager();
            if ($manager !== null && method_exists($manager, 'remove')) {
                $manager->remove($key);
            }
        }
    }

    // Додати значення в масив конфігурації
    public static function push(string $key, mixed $value): void
    {
        $array = self::get($key, []);
        if (is_array($array)) {
            $array[] = $value;
            self::set($key, $array);
        }
    }

    // Витягнути та видалити значення з конфігурації
    public static function pull(string $key, mixed $default = null): mixed
    {
        $value = self::get($key, $default);
        self::forget($key);

        return $value;
    }

    // Додати значення на початок масиву конфігурації
    public static function prepend(string $key, mixed $value): void
    {
        $array = self::get($key, []);
        if (is_array($array)) {
            array_unshift($array, $value);
            self::set($key, $array);
        }
    }
}
