<?php

/**
 * Хелпер для работы с кнопками из массивов конфигурации
 *
 * @package Flowaxy\Support\Helpers
 */

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

final class ButtonHelper
{
    /**
     * Рендеринг кнопки из массива конфигурации
     *
     * @param array<string, mixed> $config Конфигурация кнопки
     * @return string HTML кнопки
     */
    public static function render(array $config): string
    {
        $text = $config['text'] ?? 'Кнопка';
        $type = $config['type'] ?? 'primary';
        $url = $config['url'] ?? null;
        $icon = $config['icon'] ?? '';
        $attributes = $config['attributes'] ?? [];
        $submit = $config['submit'] ?? false;

        // Базовый класс кнопки
        $buttonClass = 'btn d-inline-flex align-items-center';
        if (str_starts_with($type, 'outline-')) {
            $buttonClass .= ' btn-' . $type;
        } else {
            $buttonClass .= ' btn-' . $type;
        }

        // Добавляем дополнительные классы из атрибутов
        if (isset($attributes['class'])) {
            $buttonClass .= ' ' . $attributes['class'];
            unset($attributes['class']);
        }

        // Формируем атрибуты
        $attributesString = '';
        foreach ($attributes as $key => $value) {
            if (is_bool($value) && $value) {
                $attributesString .= ' ' . htmlspecialchars($key);
            } else {
                $attributesString .= ' ' . htmlspecialchars($key) . '="' . htmlspecialchars((string)$value) . '"';
            }
        }

        // Иконка
        $iconHtml = '';
        if (!empty($icon)) {
            $iconClass = str_starts_with($icon, 'fa-') ? 'fas ' . $icon : 'fas fa-' . $icon;
            $iconHtml = '<i class="' . htmlspecialchars($iconClass) . '"></i>';
            if (!empty($text)) {
                $iconHtml .= ' ';
            }
        }

        // Если указан URL, создаем ссылку
        if ($url !== null) {
            return '<a href="' . htmlspecialchars($url) . '" class="' . htmlspecialchars($buttonClass) . '"' . $attributesString . '>' . $iconHtml . htmlspecialchars($text) . '</a>';
        }

        // Иначе создаем кнопку
        $buttonType = $submit ? 'submit' : ($attributes['type'] ?? 'button');
        return '<button type="' . htmlspecialchars($buttonType) . '" class="' . htmlspecialchars($buttonClass) . '"' . $attributesString . '>' . $iconHtml . htmlspecialchars($text) . '</button>';
    }

    /**
     * Обработка клика по кнопке (для AJAX запросов)
     *
     * @param array<string, mixed> $config Конфигурация кнопки
     * @param callable $handler Обработчик клика
     * @return string HTML кнопки с обработчиком
     */
    public static function renderWithHandler(array $config, callable $handler): string
    {
        $buttonId = $config['id'] ?? 'btn_' . uniqid();
        $config['attributes']['id'] = $buttonId;
        $config['attributes']['data-handler'] = 'true';

        // Добавляем обработчик через JavaScript
        $js = '<script>
            document.addEventListener("DOMContentLoaded", function() {
                const btn = document.getElementById("' . htmlspecialchars($buttonId) . '");
                if (btn) {
                    btn.addEventListener("click", function(e) {
                        e.preventDefault();
                        ' . self::generateHandlerJs($handler) . '
                    });
                }
            });
        </script>';

        return self::render($config) . $js;
    }

    /**
     * Генерация JavaScript кода для обработчика
     *
     * @param callable $handler Обработчик
     * @return string JavaScript код
     */
    private static function generateHandlerJs(callable $handler): string
    {
        // Простая реализация - можно расширить
        return 'console.log("Button handler called");';
    }
}
