<?php

/**
 * Хелпер для роботи з ассетами
 * Обгортка над AssetManager
 *
 * @package Flowaxy\Support\Helpers
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

use Flowaxy\Infrastructure\Assets\AssetManager;

final class AssetHelper
{
    /**
     * Додати CSS файл
     *
     * @param string $path
     * @param string|null $key
     * @return void
     */
    public static function css(string $path, ?string $key = null): void
    {
        $manager = self::getManager();
        $manager->addCss($path, $key);
    }

    /**
     * Додати JS файл
     *
     * @param string $path
     * @param string|null $key
     * @return void
     */
    public static function js(string $path, ?string $key = null): void
    {
        $manager = self::getManager();
        $manager->addJs($path, $key);
    }

    /**
     * Отримати URL зображення
     *
     * @param string $path
     * @return string
     */
    public static function image(string $path): string
    {
        return UrlHelper::uploads('images/' . ltrim($path, '/'));
    }

    /**
     * Отримати URL ассета
     *
     * @param string $path
     * @return string
     */
    public static function url(string $path): string
    {
        // Используем content/admin/resources/ для админки
        return UrlHelper::base('content/admin/resources/' . ltrim($path, '/'));
    }

    /**
     * Отримати URL з версією
     *
     * @param string $path
     * @return string
     */
    public static function version(string $path): string
    {
        // Сначала пробуем content/admin/resources/
        $fullPath = \Flowaxy\Core\System\PathResolver::contentAssets() . DS . ltrim($path, '/');

        // Fallback на старый путь
        if (!FileHelper::exists($fullPath)) {
            $fullPath = \Flowaxy\Core\System\PathResolver::admin() . DS . 'Interface' . DS . 'AdminUI' . DS . 'assets' . DS . ltrim($path, '/');
        }

        $url = self::url($path);

        if (FileHelper::exists($fullPath)) {
            $hash = md5_file($fullPath);

            return $url . '?v=' . substr($hash, 0, 8);
        }

        return $url;
    }

    /**
     * Об'єднати ассети
     *
     * @param array<int, string> $paths
     * @param string $output
     * @return bool
     */
    public static function combine(array $paths, string $output): bool
    {
        $content = '';

        foreach ($paths as $path) {
            // Сначала пробуем content/admin/resources/
            $fullPath = \Flowaxy\Core\System\PathResolver::contentAssets() . DS . ltrim($path, '/');

            // Fallback на старый путь
            if (!FileHelper::exists($fullPath)) {
                $fullPath = \Flowaxy\Core\System\PathResolver::admin() . DS . 'Interface' . DS . 'AdminUI' . DS . 'assets' . DS . ltrim($path, '/');
            }

            if (FileHelper::exists($fullPath)) {
                $content .= FileHelper::get($fullPath) . "\n";
            }
        }

        return FileHelper::put($output, $content) !== false;
    }

    /**
     * Мініфікувати ассет
     *
     * @param string $path
     * @param string|null $output
     * @return bool
     */
    public static function minify(string $path, ?string $output = null): bool
    {
        if (class_exists(\Flowaxy\Infrastructure\Assets\AssetMinifier::class)) {
            $minifier = new \Flowaxy\Infrastructure\Assets\AssetMinifier();
            $extension = FileHelper::extension($path);

            $content = FileHelper::get($path);
            if ($content === false) {
                return false;
            }

            $minified = match ($extension) {
                'css' => $minifier->minifyCss($content),
                'js' => $minifier->minifyJs($content),
                default => $content,
            };

            $output = $output ?? $path;

            return FileHelper::put($output, $minified) !== false;
        }

        return false;
    }

    /**
     * Вставити ассет inline
     *
     * @param string $path
     * @return string
     */
    public static function inline(string $path): string
    {
        // Сначала пробуем content/admin/resources/
        $fullPath = \Flowaxy\Core\System\PathResolver::contentAssets() . DS . ltrim($path, '/');

        // Fallback на старый путь
        if (!FileHelper::exists($fullPath)) {
            $fullPath = \Flowaxy\Core\System\PathResolver::admin() . DS . 'Interface' . DS . 'AdminUI' . DS . 'assets' . DS . ltrim($path, '/');
        }

        return FileHelper::get($fullPath) ?: '';
    }

    /**
     * Додати preload для ассета
     *
     * @param string $path
     * @param string $as
     * @return string
     */
    public static function preload(string $path, string $as = 'style'): string
    {
        $url = self::url($path);

        return '<link rel="preload" href="' . htmlspecialchars($url) . '" as="' . htmlspecialchars($as) . '">';
    }

    /**
     * Додати prefetch для ассета
     *
     * @param string $path
     * @return string
     */
    public static function prefetch(string $path): string
    {
        $url = self::url($path);

        return '<link rel="prefetch" href="' . htmlspecialchars($url) . '">';
    }

    /**
     * Отримати менеджер ассетів
     *
     * @return AssetManager
     */
    private static function getManager(): AssetManager
    {
        static $manager = null;

        if ($manager === null) {
            $manager = new AssetManager();
        }

        return $manager;
    }
}
