<?php

declare(strict_types=1);

namespace Flowaxy\Support\Helpers;

use function call_user_func;
use function header;
use function http_response_code;
use function is_callable;
use function json_encode;
use function strtoupper;
use const JSON_UNESCAPED_UNICODE;

// Хелпер для роботи з API endpoints з масивів конфігурації (для плагінів)
// Централізовані методи для реєстрації та обробки API маршрутів
// Оптимізовано для PHP 8.4+
final class ApiHelper
{
    // Регистрация API маршрута из массива конфигурации
    public static function registerRoute(array $route): bool
    {
        $method = strtoupper($route['method'] ?? 'GET');
        $path = $route['path'] ?? '';
        $handler = $route['handler'] ?? null;
        $auth = $route['auth'] ?? false;
        $middleware = $route['middleware'] ?? [];

        if (empty($path) || $handler === null) {
            return false;
        }

        // Реєстрація через хук (плагіни можуть використовувати)
        // @phpstan-ignore-next-line - hooks() is a global function
        \hooks()->filter('api_register_routes', function (array $routes) use ($method, $path, $handler, $auth, $middleware): array {
            $routes[] = [
                'method' => $method,
                'path' => $path,
                'handler' => $handler,
                'auth' => $auth,
                'middleware' => $middleware,
            ];
            return $routes;
        });

        return true;
    }

    // Обробка API запиту
    public static function handleRequest(string $method, string $path, array $routes): mixed
    {
        foreach ($routes as $route) {
            if (strtoupper($route['method'] ?? '') === strtoupper($method) && self::matchPath($route['path'] ?? '', $path)) {
                // Перевірка авторизації
                if (!empty($route['auth'])) {
                    if (!self::checkAuth()) {
                        return self::jsonResponse(['error' => 'Unauthorized'], 401);
                    }
                }

                // Застосування middleware
                if (!empty($route['middleware']) && is_array($route['middleware'])) {
                    foreach ($route['middleware'] as $mw) {
                        $result = self::applyMiddleware($mw);
                        if ($result !== null) {
                            return $result;
                        }
                    }
                }

                // Виклик обробника
                $handler = $route['handler'] ?? null;
                if (is_callable($handler)) {
                    return call_user_func($handler);
                }
            }
        }

        return self::jsonResponse(['error' => 'Not Found'], 404);
    }

    // Перевірка відповідності шляху маршруту
    private static function matchPath(string $routePath, string $requestPath): bool
    {
        // Просте порівняння (можна розширити для параметрів)
        return $routePath === $requestPath;
    }

    // Перевірка авторизації
    private static function checkAuth(): bool
    {
        // Проста перевірка (можна розширити)
        return \current_user_can('api.access');
    }

    // Застосування middleware
    private static function applyMiddleware(callable|string $middleware): mixed
    {
        if (is_callable($middleware)) {
            return call_user_func($middleware);
        }

        return null;
    }

    // Формування JSON відповіді
    public static function jsonResponse(mixed $data, int $statusCode = 200): string
    {
        http_response_code($statusCode);
        header('Content-Type: application/json');
        return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_THROW_ON_ERROR);
    }
}
