<?php

declare(strict_types=1);

use Flowaxy\Support\Helpers\UrlHelper;
use Flowaxy\Support\Helpers\SecurityHelper;

// Глобальные функции-хелперы для работы с админкой
// Интерфейс общения между фреймворком и контентом

// Генерация URL админки
if (!function_exists('admin_url')) {
    function admin_url(string $page = ''): string
    {
        // Используем класс UrlHelper если доступен
        if (class_exists(UrlHelper::class)) {
            return UrlHelper::admin($page);
        }

        // Fallback если класс не загружен
        $baseUrl = $_SERVER['REQUEST_SCHEME'] ?? 'http';
        $baseUrl .= '://' . ($_SERVER['HTTP_HOST'] ?? 'localhost');
        return $baseUrl . '/admin' . ($page !== '' ? '/' . ltrim($page, '/') : '');
    }
}

// Проверка разрешений текущего пользователя
// Эта функция уже определена в RoleHelper.php
// Здесь только алиас для совместимости
// Функция current_user_can уже определена в flowaxy/Support/Helpers/RoleHelper.php
// Не переопределяем её здесь, чтобы избежать конфликтов

// Проверка авторизации в админке
if (!function_exists('is_admin_logged_in')) {
    function is_admin_logged_in(): bool
    {
        if (class_exists(SecurityHelper::class)) {
            return SecurityHelper::isAdminLoggedIn();
        }

        return false;
    }
}

// Редирект на страницу админки
if (!function_exists('admin_redirect')) {
    function admin_redirect(string $page): void
    {
        // Убеждаемся, что заголовки еще не отправлены
        if (!headers_sent()) {
            $url = admin_url($page);
            header('Location: ' . $url, true, 302);
        }
        exit;
    }
}

// Получение текущей страницы админки
if (!function_exists('get_current_admin_page')) {
    function get_current_admin_page(): string
    {
        $uri = $_SERVER['REQUEST_URI'] ?? '/';

        // Извлекаем страницу из /admin/{page}
        if (preg_match('#/admin(?:/([^/?]+))?#', $uri, $matches)) {
            return $matches[1] ?? 'dashboard';
        }

        return 'dashboard';
    }
}

// Выход из админки
if (!function_exists('admin_logout')) {
    function admin_logout(): void
    {
        if (class_exists(SecurityHelper::class)) {
            SecurityHelper::logout();
        }
    }
}

// Авторизация пользователя админки
// Обертка над сервисом аутентификации
if (!function_exists('authenticate_admin_user')) {
    function authenticate_admin_user(string $username, string $password): array
    {
        // Проверяем, что SecurityHelper загружен
        if (!class_exists(SecurityHelper::class)) {
            // Пробуем загрузить через autoloader
            if (function_exists('spl_autoload_call')) {
                spl_autoload_call(SecurityHelper::class);
            }

            // Если все еще не загружен, логируем и возвращаем общую ошибку
            if (!class_exists(SecurityHelper::class)) {
                error_log("authenticate_admin_user: SecurityHelper class not found");
                try {
                    if (class_exists(\Flowaxy\Support\Facades\Log::class)) {
                        \Flowaxy\Support\Facades\Log::Error('SecurityHelper class not found in authenticate_admin_user');
                    }
                } catch (\Throwable $e) {
                    // Игнорируем ошибки логирования
                }
                return ['success' => false, 'error' => 'Невірний логін або пароль'];
            }
        }

        return SecurityHelper::authenticateAdminUser($username, $password);
    }
}

// Рендеринг layout админки
// Эта функция должна быть определена в content/admin/includes/functions.php
// Здесь не определяем, чтобы избежать конфликтов
// Функция render_admin_layout определена в content/admin/includes/functions.php
// Не определяем здесь, чтобы избежать конфликтов
