<?php

/**
 * Фасад для роботи з XML
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Filesystem\Xml as XmlInstance;
use Flowaxy\Support\Helpers\XmlHelper;

use SimpleXMLElement;

final class Xml extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return XmlInstance::class;
    }

    /**
     * Парсити XML рядок
     *
     * @param string $xml
     * @param bool $assoc
     * @return SimpleXMLElement|array<string, mixed>|false
     */
    public static function parse(string $xml, bool $assoc = true): SimpleXMLElement|array|false
    {
        return XmlHelper::parse($xml, $assoc);
    }

    /**
     * Конвертувати SimpleXMLElement в масив
     *
     * @param SimpleXMLElement $xml
     * @return array<string, mixed>
     */
    public static function toArray(SimpleXMLElement $xml): array
    {
        return XmlHelper::toArray($xml);
    }

    /**
     * Конвертувати масив в XML
     *
     * @param array<string, mixed> $array
     * @param string $rootElement
     * @return string
     */
    public static function fromArray(array $array, string $rootElement = 'root'): string
    {
        return XmlHelper::fromArray($array, $rootElement);
    }

    /**
     * Перевірити, чи рядок є валідним XML
     *
     * @param string $xml
     * @return bool
     */
    public static function validate(string $xml): bool
    {
        return XmlHelper::validate($xml);
    }

    /**
     * Прочитати XML з файлу
     *
     * @param string $path
     * @param bool $assoc
     * @return SimpleXMLElement|array<string, mixed>|false
     */
    public static function readFile(string $path, bool $assoc = true): SimpleXMLElement|array|false
    {
        return XmlHelper::readFile($path, $assoc);
    }

    /**
     * Записати XML у файл
     *
     * @param string $path
     * @param array<string, mixed>|SimpleXMLElement $data
     * @param string $rootElement
     * @return bool
     */
    public static function writeFile(string $path, array|SimpleXMLElement $data, string $rootElement = 'root'): bool
    {
        return XmlHelper::writeFile($path, $data, $rootElement);
    }

    /**
     * Отримати атрибут з XML
     *
     * @param SimpleXMLElement $xml
     * @param string $attribute
     * @return string|null
     */
    public static function getAttribute(SimpleXMLElement $xml, string $attribute): ?string
    {
        return XmlHelper::getAttribute($xml, $attribute);
    }

    /**
     * Встановити атрибут в XML
     *
     * @param SimpleXMLElement $xml
     * @param string $attribute
     * @param string $value
     * @return void
     */
    public static function setAttribute(SimpleXMLElement $xml, string $attribute, string $value): void
    {
        XmlHelper::setAttribute($xml, $attribute, $value);
    }
}
