<?php

/**
 * Фасад для роботи з завантаженням файлів
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Infrastructure\Filesystem\Upload as UploadInstance;
use Flowaxy\Support\Helpers\UploadHelper;

use function is_string;

final class Upload extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return UploadInstance::class;
    }

    /**
     * Отримати екземпляр Upload
     *
     * @param string|null $uploadDir
     * @return UploadInstance
     */
    public static function instance(?string $uploadDir = null): UploadInstance
    {
        return new UploadInstance($uploadDir);
    }

    /**
     * Обробити завантаження файлу
     *
     * @param array<string, mixed>|string $file
     * @param string|null $uploadDir
     * @return array<string, mixed>|false
     */
    public static function handle(array|string $file, ?string $uploadDir = null): array|false
    {
        $upload = new UploadInstance($uploadDir);

        if (is_string($file)) {
            $file = $_FILES[$file] ?? [];
        }

        if (empty($file) || !isset($file['tmp_name'])) {
            return false;
        }

        return $upload->handle($file);
    }

    /**
     * Встановити директорію завантаження
     *
     * @param string $uploadDir
     * @return UploadInstance
     */
    public static function setUploadDir(string $uploadDir): UploadInstance
    {
        $upload = new UploadInstance();
        return $upload->setUploadDir($uploadDir);
    }

    /**
     * Встановити дозволені розширення
     *
     * @param array<int, string> $extensions
     * @return UploadInstance
     */
    public static function setAllowedExtensions(array $extensions): UploadInstance
    {
        $upload = new UploadInstance();
        return $upload->setAllowedExtensions($extensions);
    }

    /**
     * Встановити дозволені MIME типи
     *
     * @param array<int, string> $mimeTypes
     * @return UploadInstance
     */
    public static function setAllowedMimeTypes(array $mimeTypes): UploadInstance
    {
        $upload = new UploadInstance();
        return $upload->setAllowedMimeTypes($mimeTypes);
    }

    /**
     * Встановити максимальний розмір файлу
     *
     * @param int $maxSize
     * @return UploadInstance
     */
    public static function setMaxFileSize(int $maxSize): UploadInstance
    {
        $upload = new UploadInstance();
        return $upload->setMaxFileSize($maxSize);
    }

    /**
     * Валідувати файл
     *
     * @param array<string, mixed> $file
     * @param array<int, string>|null $allowedExtensions
     * @param int|null $maxSize
     * @return bool
     */
    public static function validate(array $file, ?array $allowedExtensions = null, ?int $maxSize = null): bool
    {
        $upload = new UploadInstance();

        if ($allowedExtensions !== null) {
            $upload->setAllowedExtensions($allowedExtensions);
        }

        if ($maxSize !== null) {
            $upload->setMaxFileSize($maxSize);
        }

        return $upload->validate($file);
    }

    /**
     * Перемістити завантажений файл
     *
     * @param array<string, mixed> $file
     * @param string $destination
     * @return bool
     */
    public static function move(array $file, string $destination): bool
    {
        return UploadHelper::move($file, $destination);
    }

    /**
     * Зберегти завантажений файл
     *
     * @param array<string, mixed> $file
     * @param string $uploadDir
     * @param string|null $filename
     * @return string|false
     */
    public static function store(array $file, string $uploadDir, ?string $filename = null): string|false
    {
        return UploadHelper::store($file, $uploadDir, $filename);
    }

    /**
     * Отримати оригінальне ім'я файлу
     *
     * @param array<string, mixed> $file
     * @return string
     */
    public static function getOriginalName(array $file): string
    {
        return UploadHelper::getOriginalName($file);
    }

    /**
     * Отримати MIME тип файлу
     *
     * @param array<string, mixed> $file
     * @return string
     */
    public static function getMimeType(array $file): string
    {
        return UploadHelper::getMimeType($file);
    }

    /**
     * Отримати розмір файлу
     *
     * @param array<string, mixed> $file
     * @return int
     */
    public static function getSize(array $file): int
    {
        return UploadHelper::getSize($file);
    }

    /**
     * Перевірити, чи файл валідний
     *
     * @param array<string, mixed> $file
     * @return bool
     */
    public static function isValid(array $file): bool
    {
        return UploadHelper::isValid($file);
    }

    /**
     * Перевірити, чи файл є зображенням
     *
     * @param array<string, mixed> $file
     * @return bool
     */
    public static function isImage(array $file): bool
    {
        return UploadHelper::isImage($file);
    }
}
