<?php

/**
 * Фасад для роботи з часовими поясами
 *
 * @package Flowaxy\Support\Facades
 * @version 1.0.0 Alpha prerelease
 */

declare(strict_types=1);

namespace Flowaxy\Support\Facades;

use Flowaxy\Support\Managers\TimezoneManager;
use Flowaxy\Support\Helpers\TimezoneHelper;

use DateTime;

final class Timezone extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return TimezoneManager::class;
    }

    /**
     * Отримати менеджер часових поясів
     *
     * @return TimezoneManager
     */
    public static function manager(): TimezoneManager
    {
        return TimezoneManager::getInstance();
    }

    /**
     * Встановити часовий пояс
     *
     * @param string $timezone
     * @return bool
     */
    public static function set(string $timezone): bool
    {
        return TimezoneHelper::set($timezone);
    }

    /**
     * Отримати поточний часовий пояс
     *
     * @return string
     */
    public static function get(): string
    {
        return TimezoneHelper::get();
    }

    /**
     * Конвертувати дату в інший часовий пояс
     *
     * @param DateTime|string|int $date
     * @param string $fromTimezone
     * @param string $toTimezone
     * @return DateTime|false
     */
    public static function convert(DateTime|string|int $date, string $fromTimezone, string $toTimezone): DateTime|false
    {
        return TimezoneHelper::convert($date, $fromTimezone, $toTimezone);
    }

    /**
     * Отримати список всіх часових поясів
     *
     * @return array<int, string>
     */
    public static function list(): array
    {
        return TimezoneHelper::list();
    }

    /**
     * Перевірити, чи часовий пояс валідний
     *
     * @param string $timezone
     * @return bool
     */
    public static function isValid(string $timezone): bool
    {
        return TimezoneHelper::isValid($timezone);
    }

    /**
     * Отримати зміщення часового поясу
     *
     * @param string $timezone
     * @return int|false
     */
    public static function getOffset(string $timezone): int|false
    {
        return TimezoneHelper::getOffset($timezone);
    }

    /**
     * Отримати поточну дату та час в часовому поясі
     *
     * @param string|null $timezone
     * @return DateTime
     */
    public static function now(?string $timezone = null): DateTime
    {
        return TimezoneHelper::now($timezone);
    }

    /**
     * Форматувати дату в часовому поясі
     *
     * @param DateTime|string|int $date
     * @param string $format
     * @param string|null $timezone
     * @return string|false
     */
    public static function format(DateTime|string|int $date, string $format = 'Y-m-d H:i:s', ?string $timezone = null): string|false
    {
        return TimezoneHelper::format($date, $format, $timezone);
    }
}
